/**********************************************************************
 Copyright (c) 2005 Andy Jefferson and others. All rights reserved.
 Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License. 
 

 Contributors:
 ...
 **********************************************************************/
package org.datanucleus.store.mapped;

import org.datanucleus.store.mapped.mapping.JavaTypeMapping;

/**
 * Representation of a statement to fetch a particular object.
 */
public interface FetchStatement
{
    /**
     * Add column(s) to the SELECT clause using a LEFT OUTER JOIN
     * to join to another table. Joins using the refMapping in the other table,
     * and selects the columns of the selectMapping. A discriminator in the other table
     * can be provided where this should be part of the join constraint.
     * @param refMapping Mapping in the other table to join to.
     * @param selectMapping Mapping of the field in the other table to select
     * @param refDiscrimMapping Mapping for discriminator in the related table (optional)
     * @param discrimValues Valid values for the discriminator in the related table (optional)
     * @param useExistingJoinIfPossible Use an existing outer join if possible
     * @return The parameter position(s) of the column(s) in the statement.
     */
    public abstract int[] selectOuterJoin(JavaTypeMapping refMapping, JavaTypeMapping selectMapping,
            JavaTypeMapping refDiscrimMapping, Object[] discrimValues, boolean useExistingJoinIfPossible);

    /**
     * Add column(s) to the SELECT clause using a LEFT OUTER JOIN
     * to join to another table. Joins using the refMapping in the other table,
     * and selects the columns of the selectMapping. A reference type mapping can be provided
     * where we need the reference mapping joining to a subclass table to define the
     * correct type.
     * @param refMapping Mapping in the other table to join to.
     * @param selectMapping Mapping of the field in the other table to select
     * @param refTypeMapping Mapping for subclass table for the related table (optional)
     * @return The parameter position(s) of the column(s) in the statement.
     */
    public abstract int[] selectOuterJoin(JavaTypeMapping refMapping, JavaTypeMapping selectMapping,
            JavaTypeMapping refTypeMapping);

    /**
     * Add a column to the SELECT clause.
     * @param col The column to add
     * @return The parameter position of this column in the statement.
     */
    public abstract int select(DatastoreField col);

    /**
     * Adds an expression to the SELECT clause.
     * @param expr the expr to add to the select clause
     * @return The parameter position of this expression in the statement.
     */
    public abstract int select(String expr);

    /**
     * Method to specify a column to be referenced.
     * @param col The column
     * @return The column statement text
     */
    public abstract String referenceDatastoreField(DatastoreField col);

    /**
     * Method to add an extra condition to the WHERE clause.
     * @param condition The condition
     */
    public abstract void andCondition(String condition);

    /**
     * Method to return the statement assuming no locking of rows.
     * @return The statement
     */
    public abstract String toString();

    /**
     * Method to return the statement.
     * @param lock Whether to lock the rows found by this SELECT
     * @return The statement
     **/
    public abstract String toString(boolean lock);
}