/**********************************************************************
Copyright (c) 2012 Andy Jefferson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
   ...
**********************************************************************/
package org.datanucleus.management;

import org.datanucleus.util.MathUtils;

/**
 * Abstract base class for a statistics object.
 */
public abstract class AbstractStatistics
{
    AbstractStatistics parent = null;
    
    String registeredName;

    int numReads = 0;
    int numWrites = 0;
    int numReadsLastTxn = 0;
    int numWritesLastTxn = 0;

    int numReadsStartTxn = 0; // Work variable
    int numWritesStartTxn = 0; // Work variable

    int insertCount = 0;
    int deleteCount = 0;
    int updateCount = 0;
    int fetchCount = 0;

    int txnTotalCount;
    int txnCommittedTotalCount;
    int txnRolledBackTotalCount;
    int txnActiveTotalCount;
    int txnExecutionTotalTime = 0;
    int txnExecutionTimeHigh =-1;
    int txnExecutionTimeLow =-1;
    MathUtils.SMA txnExecutionTimeAverage = new MathUtils.SMA(50);

    int queryActiveTotalCount;
    int queryErrorTotalCount;
    int queryExecutionTotalCount;
    int queryExecutionTotalTime = 0;
    int queryExecutionTimeHigh =-1;
    int queryExecutionTimeLow =-1;
    MathUtils.SMA queryExecutionTimeAverage = new MathUtils.SMA(50);

    /**
     * Constructor specifying a "name" that we want to know this by.
     * The name is simply a JMX MBean name if using JMX, or null otherwise.
     * @param name Name that is known by
     */
    public AbstractStatistics(String name)
    {
        this.registeredName = name;
    }

    public String getRegisteredName()
    {
        return registeredName;
    }

    public int getQueryActiveTotalCount()
    {
        return queryActiveTotalCount;
    }

    public int getQueryErrorTotalCount()
    {
        return queryErrorTotalCount;
    }

    public int getQueryExecutionTotalCount()
    {
        return queryExecutionTotalCount;
    }

    public int getQueryExecutionTimeLow()
    {
        return queryExecutionTimeLow;
    }

    public int getQueryExecutionTimeHigh()
    {
        return queryExecutionTimeHigh;
    }

    public int getQueryExecutionTotalTime()
    {
        return queryExecutionTotalTime;
    }

    public int getQueryExecutionTimeAverage()
    {
        return (int) queryExecutionTimeAverage.currentAverage();
    }

    public void queryBegin()
    {
        this.queryActiveTotalCount++;
        if (parent != null)
        {
            parent.queryBegin();
        }
    }

    public void queryExecutedWithError()
    {
        this.queryErrorTotalCount++;
        this.queryActiveTotalCount--;
        if (parent != null)
        {
            parent.queryExecutedWithError();
        }
    }
    
    public void queryExecuted(long executionTime)
    {
        this.queryExecutionTotalCount++;
        this.queryActiveTotalCount--;
        queryExecutionTimeAverage.compute(executionTime);
        queryExecutionTimeLow = (int) Math.min(queryExecutionTimeLow==-1?executionTime:queryExecutionTimeLow,executionTime);
        queryExecutionTimeHigh = (int) Math.max(queryExecutionTimeHigh,executionTime);
        queryExecutionTotalTime += executionTime;
        if (parent != null)
        {
            parent.queryExecuted(executionTime);
        }
    }

    public int getNumberOfDatastoreWrites()
    {
        return numWrites;
    }

    public int getNumberOfDatastoreReads()
    {
        return numReads;
    }

    public int getNumberOfDatastoreWritesInLatestTxn()
    {
        return numWritesLastTxn;
    }

    public int getNumberOfDatastoreReadsInLatestTxn()
    {
        return numReadsLastTxn;
    }

    public void incrementNumReads()
    {
        numReads++;
        if (parent != null)
        {
            parent.incrementNumReads();
        }
    }

    public void incrementNumWrites()
    {
        numWrites++;
        if (parent != null)
        {
            parent.incrementNumWrites();
        }
    }

    public int getNumberOfObjectFetches()
    {
        return fetchCount;
    }

    public int getNumberOfObjectInserts()
    {
        return insertCount;
    }

    public int getNumberOfObjectUpdates()
    {
        return updateCount;
    }

    public int getNumberOfObjectDeletes()
    {
        return deleteCount;
    }

    public void incrementInsertCount()
    {
        insertCount++;
        if (parent != null)
        {
            parent.incrementInsertCount();
        }
    }

    public void incrementDeleteCount()
    {
        deleteCount++;
        if (parent != null)
        {
            parent.incrementDeleteCount();
        }
    }

    public void incrementFetchCount()
    {
        fetchCount++;
        if (parent != null)
        {
            parent.incrementFetchCount();
        }
    }

    public void incrementUpdateCount()
    {
        updateCount++;
        if (parent != null)
        {
            parent.incrementUpdateCount();
        }
    }

    public int getTransactionExecutionTimeAverage()
    {
        return (int) txnExecutionTimeAverage.currentAverage();
    }

    public int getTransactionExecutionTimeLow()
    {
        return txnExecutionTimeLow;
    }

    public int getTransactionExecutionTimeHigh()
    {
        return txnExecutionTimeHigh;
    }

    public int getTransactionExecutionTotalTime()
    {
        return txnExecutionTotalTime;
    }

    public int getTransactionTotalCount()
    {
        return txnTotalCount;
    }
    
    public int getTransactionActiveTotalCount()
    {
        return txnActiveTotalCount;
    }

    public int getTransactionCommittedTotalCount()
    {
        return txnCommittedTotalCount;
    }

    public int getTransactionRolledBackTotalCount()
    {
        return txnRolledBackTotalCount;
    }
    
    public void transactionCommitted(long executionTime)
    {
        this.txnCommittedTotalCount++;
        this.txnActiveTotalCount--;
        txnExecutionTimeAverage.compute(executionTime);
        txnExecutionTimeLow = (int) Math.min(txnExecutionTimeLow==-1?executionTime:txnExecutionTimeLow,executionTime);
        txnExecutionTimeHigh = (int) Math.max(txnExecutionTimeHigh,executionTime);
        txnExecutionTotalTime += executionTime;

        numReadsLastTxn = numReads - numReadsStartTxn;
        numWritesLastTxn = numWrites - numWritesStartTxn;
        if (parent != null)
        {
            parent.transactionCommitted(executionTime);
        }
    }
    
    public void transactionRolledBack(long executionTime)
    {
        this.txnRolledBackTotalCount++;
        this.txnActiveTotalCount--;
        txnExecutionTimeAverage.compute(executionTime);
        txnExecutionTimeLow = (int) Math.min(txnExecutionTimeLow==-1?executionTime:txnExecutionTimeLow,executionTime);
        txnExecutionTimeHigh = (int) Math.max(txnExecutionTimeHigh,executionTime);
        txnExecutionTotalTime += executionTime;

        numReadsLastTxn = numReads - numReadsStartTxn;
        numWritesLastTxn = numWrites - numWritesStartTxn;
        if (parent != null)
        {
            parent.transactionRolledBack(executionTime);
        }
    }
    
    public void transactionStarted()
    {
        this.txnTotalCount++;
        this.txnActiveTotalCount++;

        numReadsStartTxn = numReads;
        numWritesStartTxn = numWrites;
        if (parent != null)
        {
            parent.transactionStarted();
        }
    }
}