// Copyright 2010 Google Inc. All Rights Reserved.

package com.google.appengine.api;

import com.google.apphosting.api.ApiProxy;
import java.security.AccessController;
import java.security.PrivilegedAction;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Information about the current status of the Java Runtime.
 *
 */
public final class LifecycleManager {

  private static final LifecycleManager instance = new LifecycleManager();

  private volatile boolean shuttingDown = false;
  private volatile long deadline = -1L;

  private LifecycleManager() { }

  public static LifecycleManager getInstance() {
    return instance;
  }

  public boolean isShuttingDown() {
    return shuttingDown;
  }

  /**
   * Register a ShutdownHook to be called when the runtime shuts down.
   *
   * @throws NullPointerException if the calling thread is neither a request thread nor a thread
   *     created by {@link com.google.appengine.api.ThreadManager ThreadManager}.
   */
  public synchronized void setShutdownHook(ShutdownHook hook) {
    hooks.put(currentAppVersionId(), hook);
  }

  /**
   * Calls Thread.interrupt() on all threads running requests for this
   * application.
   */
  public void interruptAllRequests() {
    AccessController.doPrivileged(
        new PrivilegedAction<Void>() {
          @Override public Void run() {
            List<Thread> threads = ApiProxy.getRequestThreads();
            if (threads != null) {
              for (Thread thread : threads) {
                thread.interrupt();
              }
            }
            return null;
          }
        });
  }

  /**
   * If the runtime is shutting down, returns how long, in
   * milliseconds, is left for shutdown code to clean up. Otherwise,
   * returns -1.
   */
  public long getRemainingShutdownTime() {
    long value = deadline;
    if (value == -1L) {
      return -1L;
    } else {
      return value - System.currentTimeMillis();
    }
  }

  /**
   * For testing purposes only:
   * Notifies the LifecycleManager that the runtime is shutting down.
   */
  public synchronized void beginShutdown(long deadline) {
    this.shuttingDown = true;
    this.deadline = deadline;
    ShutdownHook hook = hooks.get(currentAppVersionId());
    if (hook != null) {
      hook.shutdown();
    }
  }

  private String currentAppVersionId() {
    ApiProxy.Environment env = ApiProxy.getCurrentEnvironment();
    if (env == null) {
      throw new NullPointerException(
          "Operation not allowed in a thread that is neither the original request thread "
              + "nor a thread created by ThreadManager");
    }
    return env.getAppId() + "/" + env.getVersionId();
  }

  private Map<String, ShutdownHook> hooks = new HashMap<String, ShutdownHook>();

  public interface ShutdownHook {
    void shutdown();
  }
}
