/**********************************************************************
Copyright (c) 2006 Andy Jefferson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
    ...
**********************************************************************/
package org.datanucleus.store.mapped.mapping;

import org.datanucleus.ClassNameConstants;
import org.datanucleus.ObjectManager;
import org.datanucleus.store.mapped.expression.IntegerLiteral;
import org.datanucleus.store.mapped.expression.LogicSetExpression;
import org.datanucleus.store.mapped.expression.NumericExpression;
import org.datanucleus.store.mapped.expression.QueryExpression;
import org.datanucleus.store.mapped.expression.ScalarExpression;

/**
 * Abstract SCO mapping for a java type that will be stored as a Long type.
 */
public abstract class ObjectAsLongMapping extends SingleFieldMapping
{
    /**
     * Method to create a literal of this type for use in a JDOQL query.
     * @param qs The QueryExpression
     * @param value The value of the literal
     * @return The Literal
     */
    public ScalarExpression newLiteral(QueryExpression qs, Object value)
    {
        return new IntegerLiteral(qs, this, objectToLong(value));
    }

    /**
     * Method to create a new expression for this mapping for use in a JDOQL Query.
     * @param qs The QueryExpression
     * @param te Expression for the datastore container
     * @return The JDOQL expression
     */
    public ScalarExpression newScalarExpression(QueryExpression qs, LogicSetExpression te)
    {
        return new NumericExpression(qs, this, te);
    }

    /**
     * Method to return the Java type.
     * @return The Java type being represented.
     */
    public abstract Class getJavaType();

    /**
     * Accessor for the name of the java-type actually used when mapping the particular datastore
     * field. This java-type must have an entry in the datastore mappings.
     * @param index requested datastore field index.
     * @return the name of java-type for the requested datastore field.
     */
    public String getJavaTypeForDatastoreMapping(int index)
    {
        // All of the types extending this class will be using java-type of Long for the datastore
        return ClassNameConstants.JAVA_LANG_LONG;
    }

    /**
     * Method to set the object when updating the the datastore.
     * @see org.datanucleus.store.mapped.mapping.SingleFieldMapping#setObject(org.datanucleus.ObjectManager, java.lang.Object, int[], java.lang.Object)
     */
    public void setObject(ObjectManager om, Object preparedStatement, int[] exprIndex, Object value)
    {
        getDataStoreMapping(0).setObject(preparedStatement, exprIndex[0], objectToLong(value));
    }

    /**
     * Method to get the object from the datastore and convert to an object.
     * @see org.datanucleus.store.mapped.mapping.SingleFieldMapping#getObject(org.datanucleus.ObjectManager, java.lang.Object, int[])
     */
    public Object getObject(ObjectManager om, Object resultSet, int[] exprIndex)
    {
        if (exprIndex == null)
        {
            return null;
        }

        Object datastoreValue = getDataStoreMapping(0).getObject(resultSet, exprIndex[0]);
        Object value = null;
        if (datastoreValue != null)
        {
            value = longToObject((Long)datastoreValue);
        }
        return value;
    }

    /**
     * Method to set the datastore value based on the object value.
     * @param object The object
     * @return The value to pass to the datastore
     */
    protected abstract Long objectToLong(Object object);

    /**
     * Method to extract the objects value from the datastore object.
     * @param datastoreValue Value obtained from the datastore
     * @return The value of this object (derived from the datastore value)
     */
    protected abstract Object longToObject(Long datastoreValue);
}