/**********************************************************************
Copyright (c) 2005 Andy Jefferson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License. 


Contributors:
    ...
**********************************************************************/
package org.datanucleus.store.mapped.mapping;

import org.datanucleus.ObjectManager;
import org.datanucleus.StateManager;
import org.datanucleus.state.StateManagerFactory;

/**
 * Mapping for a serialised PersistenceCapable object.
 * Extends ObjectMapping since that provides the basic serialisation mechanism,
 * adding on the addition of StateManagers to the serialised object whenever it is required.
 */
public class SerialisedPCMapping extends SerialisedMapping
{
    /**
     * Method to populate parameter positions in a PreparedStatement with this object
     * @param om The Object Manager
     * @param preparedStatement The Prepared Statement
     * @param exprIndex The parameter positions to populate
     * @param value The value of the PC to use in populating the parameter positions
     */
    public void setObject(ObjectManager om, Object preparedStatement, int[] exprIndex, Object value)
    {
        setObject(om, preparedStatement, exprIndex, value, null, mmd.getAbsoluteFieldNumber());
    }

    /**
     * Method to populate parameter positions in a PreparedStatement with this object
     * @param om The Object Manager
     * @param preparedStatement The Prepared Statement
     * @param exprIndex The parameter positions to populate
     * @param value The value of the PC to use in populating the parameter positions
     * @param ownerSM State Manager for the owning object
     * @param fieldNumber field number of this object in the owning object
     */
    public void setObject(ObjectManager om, Object preparedStatement, int[] exprIndex, Object value, 
            StateManager ownerSM, int fieldNumber)
    {
        if (value != null)
        {
            // Assign a StateManager to the serialised object if none present
            StateManager embSM = om.findStateManager(value);
            if (embSM == null || om.getApiAdapter().getObjectManager(value) == null)
            {
                embSM = StateManagerFactory.newStateManagerForEmbedded(om, value, false);
                embSM.addEmbeddedOwner(ownerSM, fieldNumber);
            }
        }

        StateManager sm = null;
        if (value != null)
        {
            // Find SM for serialised object
            sm = om.findStateManager(value);
        }

        if (sm != null)
        {
            sm.setStoringPC();
        }
        getDataStoreMapping(0).setObject(preparedStatement, exprIndex[0], value);
        if (sm != null)
        {
            sm.unsetStoringPC();
        }
    }

    /**
     * Method to extract the value of the PersistenceCapable from a ResultSet.
     * @param om The ObjectManager
     * @param resultSet The ResultSet
     * @param exprIndex The parameter positions in the result set to use.
     * @return The (deserialised) PersistenceCapable object
     */
    public Object getObject(ObjectManager om, Object resultSet, int[] exprIndex)
    {
        return getObject(om, resultSet, exprIndex, null, mmd.getAbsoluteFieldNumber());
    }

    /**
     * Method to extract the value of the PersistenceCapable from a ResultSet.
     * @param om The ObjectManager
     * @param resultSet The ResultSet
     * @param exprIndex The parameter positions in the result set to use.
     * @param ownerSM The owning object
     * @param fieldNumber Absolute number of field in owner object
     * @return The (deserialised) PersistenceCapable object
     */
    public Object getObject(ObjectManager om, Object resultSet, int[] exprIndex, StateManager ownerSM, int fieldNumber)
    {
        Object obj = getDataStoreMapping(0).getObject(resultSet, exprIndex[0]);
        if (obj != null)
        {
            // Assign a StateManager to the serialised object if none present
            StateManager embSM = om.findStateManager(obj);
            if (embSM == null || om.getApiAdapter().getObjectManager(obj) == null)
            {
                embSM = StateManagerFactory.newStateManagerForEmbedded(om, obj, false);
                embSM.addEmbeddedOwner(ownerSM, fieldNumber); // Feed dirty flags to the owning object
            }
        }
        return obj;
    }
}