/**********************************************************************
Copyright (c) 2006 Andy Jefferson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
    ...
**********************************************************************/
package org.datanucleus.metadata.annotations;

import java.lang.annotation.Annotation;
import java.lang.reflect.Constructor;
import java.util.HashMap;
import java.util.Map;

import org.datanucleus.ClassLoaderResolver;
import org.datanucleus.ObjectManagerFactoryImpl;
import org.datanucleus.metadata.AbstractClassMetaData;
import org.datanucleus.metadata.MetaDataManager;
import org.datanucleus.metadata.PackageMetaData;
import org.datanucleus.metadata.annotations.AnnotationManager;
import org.datanucleus.plugin.ConfigurationElement;
import org.datanucleus.plugin.Extension;
import org.datanucleus.util.NucleusLogger;
import org.datanucleus.util.Localiser;

/**
 * Manager for annotations.
 * Acts as a registry of the available annotation readers and allows use of all
 * types of registered annotations.
 */
public class AnnotationManagerImpl implements AnnotationManager
{
    /** Localiser for messages */
    protected static Localiser LOCALISER=Localiser.getInstance(
        "org.datanucleus.Localisation", ObjectManagerFactoryImpl.class.getClassLoader());

    /** MetaData Manager that we work for. */
    protected final MetaDataManager metadataMgr;

    /** Lookup of annotation reader name keyed by the annotation class name. */
    Map<String, String> annotationReaderLookup = new HashMap<String, String>();

    /** Cache of the available annotation readers (keyed by the class name). */
    Map<String, AnnotationReader> annotationReaders = new HashMap<String, AnnotationReader>();

    /**
     * Constructor.
     * @param metadataMgr Manager for MetaData
     */
    public AnnotationManagerImpl(MetaDataManager metadataMgr)
    {
        this.metadataMgr = metadataMgr;

        // Load up the registry of available annotation readers
        Extension[] ex = this.metadataMgr.getOMFContext().getPluginManager().getExtensionPoint(
            "org.datanucleus.annotations").getExtensions();
        for (int i=0; i<ex.length; i++)
        {
            ConfigurationElement[] confElm = ex[i].getConfigurationElements();
            for (int j=0; j<confElm.length; j++)
            {
                annotationReaderLookup.put(confElm[j].getAttribute("annotation-class"), confElm[j].getAttribute("reader"));
            }
        }
    }

    /**
     * Accessor for the MetaData for the specified class, read from annotations.
     * The annotations can be of any supported type.
     * @param cls The class
     * @param pmd PackageMetaData to use as a parent
     * @param clr ClassLoader resolver
     * @return The ClassMetaData
     */
    public AbstractClassMetaData getMetaDataForClass(Class cls, PackageMetaData pmd, ClassLoaderResolver clr)
    {
        if (cls == null)
        {
            return null;
        }

        Annotation[] annotations = cls.getAnnotations();
        if (annotations == null || annotations.length == 0)
        {
            return null;
        }

        // Find an annotation reader for this classes annotations (if we have one)
        String readerClassName = null;
        for (int i=0;i<annotations.length;i++)
        {
            String reader = annotationReaderLookup.get(annotations[i].annotationType().getName());
            if (reader != null)
            {
                readerClassName = reader;
                break;
            }
        }
        if (readerClassName == null)
        {
            NucleusLogger.METADATA.debug(LOCALISER.msg("044202", cls.getName()));
            return null;
        }

        // Check if we have a reader of this type
        AnnotationReader reader = annotationReaders.get(readerClassName);
        if (reader == null)
        {
            try
            {
                Class annotationReaderClass = clr.classForName(readerClassName,ObjectManagerFactoryImpl.class.getClassLoader());

                Class[] ctrArgs = new Class[] {MetaDataManager.class};
                Object[] ctrParams = new Object[] {metadataMgr};
                Constructor ctor = annotationReaderClass.getConstructor(ctrArgs);
                reader = (AnnotationReader)ctor.newInstance(ctrParams);
                annotationReaders.put(readerClassName, reader); // Save the annotation reader in case we have more of this type
            }
            catch (Exception e)
            {
                NucleusLogger.METADATA.warn(LOCALISER.msg("MetaData.AnnotationReaderNotFound", readerClassName));
                return null;
            }
        }

        return reader.getMetaDataForClass(cls, pmd, clr);
    }
}
