/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the reusable ccl java library
 * (http://www.kclee.com/clemens/java/ccl/).
 *
 * The Initial Developer of the Original Code is
 * Chr. Clemens Lee.
 * Portions created by Chr. Clemens Lee are Copyright (C) 2002
 * Chr. Clemens Lee. All Rights Reserved.
 *
 * Contributor(s): Chr. Clemens Lee <clemens@kclee.com>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

package ccl.servlet;

import ccl.util.Util;
import java.io.IOException;
import java.io.PrintWriter;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import javax.servlet.http.HttpServletResponse;

/**
 * This class encapsulates printing of spread sheet csv
 * data generated by a servlet.
 *
 * @author    Chr. Clemens Lee (clemens@kclee.com)
 * @version   $Id: CSV.java,v 1.5 2003/05/01 16:44:02 clemens Exp clemens $
 */
public class CSV 
{
    private PrintWriter _out = null;

    /**
     * All following output goes to the given
     * HttpServletResponse object. The content type is
     * set immediately to text/plain.
     * Use the close method on this object to finish the
     * CSV output.
     */
    public CSV( HttpServletResponse response ) 
        throws IOException
    {
        super();

        //response.setContentType( "application/vnd.ms-excel" );
        response.setContentType( "text/plain" );
        _out = response.getWriter();
        _bFirstColumn = true;
    }

    private boolean _bFirstColumn = true;

    /**
     * This method ends a row. Next data will appear in a new
     * line.
     */
    public void endLine()
    {
        _out.print( "\n" );     // better "\r\n" ?
        _bFirstColumn = true;
    }

    /**
     * Fills data into one field.
     */
    public void printValue( String value )
    {
        if ( !_bFirstColumn )
        {
            _out.print( "\t" );
        }
        _out.print( Util.replace( value, "\t", " " ) );
        _bFirstColumn = false;
    }

    /**
     * This method just closes any open stream.
     * There should be no following invokation on this object.
     */
    public void close() 
    {
        _out.close();
    }

    /**
     * Convert all tabs to space (1).
     */
    static private String untabify( String s )
    {
        return Util.replace( s, "\t", " " );
    }

    /**
     * This method does not print html output but instead
     * creates a csv file which can be used to import into
     * a spread sheet application like Excel or StarCalc.
     * The data is taken from the given result set.
     *
     * @exception   IOException    in case anything goes wrong
     *                             writing output to the http
     *                             servlet response.
     * @exception   SQLException   in case anything goes wrong
     *                             with the database.
     */
    static public void exportCSV( HttpServletResponse response
                                  , ResultSet result )
        throws IOException
               , SQLException
    {
        response.setContentType( "text/plain" );
        PrintWriter out = response.getWriter();

        ResultSetMetaData metaData = result.getMetaData();
        int numberOfColumns = metaData.getColumnCount();

        // header line
        for( int column = 1; column <= numberOfColumns; column++ ) 
        {
            String sColumnName = metaData.getColumnName( column );
            if ( column > 1 )
            {
                out.print( "\t" );
            }
            out.print( untabify( sColumnName ) );
        }
        out.println();

        while( result.next() )
        {
            for( int column = 1; column <= numberOfColumns; column++ ) 
            {
                Object oValue = result.getObject( column );
                if ( oValue == null )
                {
                    oValue = (Object)"";
                }
                String sValue = untabify( String.valueOf( oValue ).trim() );
                if ( column > 1 )
                {
                    out.print( "\t" );
                }
                out.print( sValue );
            }
            out.println();
        }
    }
}
