/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the reusable ccl java library
 * (http://www.kclee.com/clemens/java/ccl/).
 *
 * The Initial Developer of the Original Code is
 * Chr. Clemens Lee.
 * Portions created by Chr. Clemens Lee are Copyright (C) 2002
 * Chr. Clemens Lee. All Rights Reserved.
 *
 * Contributor(s): Chr. Clemens Lee <clemens@kclee.com>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

package ccl.swing;

import ccl.util.Util;
import java.awt.Component;
import java.awt.Container;
import java.awt.GridBagLayout;
import java.awt.GridBagConstraints;
import java.awt.Insets;

/**
 * This layout manager hides a lot of complexity from the
 * GridBagLayout manager, especially through hiding the
 * GridBagConstraints object. Meanwhile you can still do
 * all the things the original GridBagLayout manager offers.
 *
 * Personally This is one of the two most regularly used
 * classes of the ccl.swing package (together with the
 * ccl.swing.SwingUtil class of course).
 *
 * @version  $Id: AutoGridBagLayout.java,v 1.13 2003/05/01 16:44:08 clemens Exp clemens $
 * @author <a href="http://www.kclee.com/clemens/">
 *         Chr. Clemens Lee</a>
 *         &lt;<a href="mailto:clemens@kclee.com">
 *         clemens@kclee.com
 *         </a>>
 */
public class AutoGridBagLayout extends GridBagLayout 
{
    /**
     * Put the component in the center of its display area.
     */
    public static final int CENTER = GridBagConstraints.CENTER;

    /**
     * Put the component at the top of its display area,
     * centered horizontally. 
     */
    public static final int NORTH = GridBagConstraints.NORTH;

    /**
     * Put the component at the top-right corner of its display area. 
     */
    public static final int NORTHEAST = GridBagConstraints.NORTHEAST;

    /**
     * Put the component on the right side of its display area, 
     * centered vertically.
     */
    public static final int EAST = GridBagConstraints.EAST;

    /**
     * Put the component at the bottom-right corner of its display area. 
     */
    public static final int SOUTHEAST = GridBagConstraints.SOUTHEAST;

    /**
     * Put the component at the bottom of its display area, centered 
     * horizontally. 
     */
    public static final int SOUTH = GridBagConstraints.SOUTH;

    /**
     * Put the component at the bottom-left corner of its display area. 
     */
    public static final int SOUTHWEST = GridBagConstraints.SOUTHWEST;

    /**
     * Put the component on the left side of its display area, 
     * centered vertically.
     */
    public static final int WEST = GridBagConstraints.WEST;

    /**
     * Put the component at the top-left corner of its display area. 
     */
    public static final int NORTHWEST = GridBagConstraints.NORTHWEST;

    private GridBagConstraints _pGridBagConstraints = null;

    private int _x = 0;
    private int _y = 0;

    private int _extendX = 1;
    private int _extendY = 1;

    public AutoGridBagLayout() 
    {
        this( 2 );
    }

    public AutoGridBagLayout( int insets_ ) 
    {
        super();
        _pGridBagConstraints = new GridBagConstraints();
        setInsets( insets_ );
        setAnchor(GridBagConstraints.CENTER);
        setFillNone();
        _pGridBagConstraints.gridx = _x;
        _pGridBagConstraints.gridy = _y;
        _pGridBagConstraints.gridwidth = 1;
        _pGridBagConstraints.gridheight = 1;
        _pGridBagConstraints.weightx = 0.0;
        _pGridBagConstraints.weighty = 0.0;
    }

    public void setInsets(int inset) 
    {
        setInsets(new Insets(inset, inset, inset, inset));
    }

    public void setInsets(int vertical_, int horizontal_) 
    {
        setInsets(new Insets(vertical_, horizontal_, vertical_, horizontal_));
    }

    public void setInsets(Insets pInsets_) 
    {
        _pGridBagConstraints.insets = (Insets) pInsets_.clone();
    }

    public void setAnchor(int anchor) 
    {
        _pGridBagConstraints.anchor = anchor;
    }

    public void setAnchorNorth() 
    {
        setAnchor(GridBagConstraints.NORTH);
    }
    
    public void setAnchorEast() 
    {
        setAnchor(GridBagConstraints.EAST);
    }

    public void setAnchorSouth() 
    {
        setAnchor(GridBagConstraints.SOUTH);
    }

    public void setAnchorWest() 
    {
        setAnchor(GridBagConstraints.WEST);
    }

    public void setAnchorCenter() 
    {
        setAnchor(GridBagConstraints.CENTER);
    }

    public void setFillNone() 
    {
        _pGridBagConstraints.fill = GridBagConstraints.NONE;
    }

    public void setExpandNone() 
    {
        _pGridBagConstraints.weightx = 0.0;
        _pGridBagConstraints.weighty = 0.0;
    }

    public void setFillBoth() 
    {
        _pGridBagConstraints.fill = GridBagConstraints.BOTH;
    }

    /**
     * Specifies that the next component to be added
     * should be expanded horizontally and vertically
     * with a default weight of 1.0 in each direction.
     */
    public void setExpandBoth() 
    {
        _pGridBagConstraints.weightx = 1.0;
        _pGridBagConstraints.weighty = 1.0;
    }

    /**
     * Sets the weight used for the next component to expand
     * its space. Standard values resolve aroudn 1.0.
     */
    public void setExpandBoth(double dX_, double dY_) 
    {
        _pGridBagConstraints.weightx = dX_;
        _pGridBagConstraints.weighty = dY_;
    }

    public void setFillHorizontal() 
    {
        _pGridBagConstraints.fill = GridBagConstraints.HORIZONTAL;
    }

    public void setExpandHorizontal() 
    {
        _pGridBagConstraints.weightx = 1.0;
        _pGridBagConstraints.weighty = 0.0;
    }

    public void setExpandHorizontal(double dHorizontal_) 
    {
        _pGridBagConstraints.weightx = dHorizontal_;
        _pGridBagConstraints.weighty = 0.0;
    }
    
    public void setFillVertical() 
    {
        _pGridBagConstraints.fill = GridBagConstraints.VERTICAL;
    }

    public void setExpandVertical() 
    {
        _pGridBagConstraints.weightx = 0.0;
        _pGridBagConstraints.weighty = 1.0;
    }

    public void setExpandVertical(double dVertical_) 
    {
        _pGridBagConstraints.weightx = 0.0;
        _pGridBagConstraints.weighty = dVertical_;
    }

    public void skip() 
    {
        _x += _extendX;
    }

    public void endLine() 
    {
        _x = 0;
        _y += _extendY;
    }

    public void nextLine() 
    {
        _x -= _extendX;
        _y += _extendY;
    }

    public void gotoGrid(int x_, int y_) 
    {
        _x = x_;
        _y = y_;
    }

    public void setExtend(int extendX_, int extendY_) 
    {
        _extendX = extendX_;
        _extendY = extendY_;
        _pGridBagConstraints.gridwidth = _extendX;
        _pGridBagConstraints.gridheight = _extendY;
    }

    /**
     * Adds the specified component to the layout, using the specified
     * constraint object.
     * @param   pComponent_     the component to be added
     * @param   oConstraints_   where/how the component is added to the layout.
     */
    public void addLayoutComponent(Component pComponent_, Object oConstraints_)
    {
        Util.panicIf(oConstraints_ != null);
        
        Util.debug( this
                    , "addLayoutComponent(..).x,y: " + _x + "," + _y );
        
        _pGridBagConstraints.gridx = _x;
        _pGridBagConstraints.gridy = _y;
        setConstraints(pComponent_, _pGridBagConstraints);
        _x += _extendX;
    }

    public void setLayoutOn( Container pContainer_ ) 
    {
        pContainer_.setLayout( this );
    }

    public void add( Container container_, 
                     Component component_ ) 
    {
        container_.add( component_ );
    }
}
