/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the reusable ccl java library
 * (http://www.kclee.com/clemens/java/ccl/).
 *
 * The Initial Developer of the Original Code is
 * Chr. Clemens Lee.
 * Portions created by Chr. Clemens Lee are Copyright (C) 2002
 * Chr. Clemens Lee. All Rights Reserved.
 *
 * Contributor(s): Chr. Clemens Lee <clemens@kclee.com>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

package ccl.swing;

import ccl.util.FileUtil;
import ccl.util.Util;
import com.sun.image.codec.jpeg.JPEGCodec;
import com.sun.image.codec.jpeg.JPEGEncodeParam;
import com.sun.image.codec.jpeg.JPEGImageEncoder;
import java.awt.Component;
import java.awt.Container;
import java.awt.Graphics2D;
import java.awt.Image;
import java.awt.Rectangle;
import java.awt.geom.AffineTransform;
import java.awt.image.BufferedImage;
import java.awt.image.DataBuffer;
import java.awt.image.Raster;
import java.awt.image.SampleModel;
import java.io.BufferedOutputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import javax.swing.ImageIcon;
import javax.swing.SwingUtilities;
import net.jmge.gif.Gif89Encoder;

/**
 * Support class for graphics stuff, like for example saving jpeg images.
 * 
 * @author  Chr. Clemens Lee
 * @version $Id: GraphicsUtil.java,v 1.5 2003/10/18 07:53:16 clemens Exp clemens $
 */
public class GraphicsUtil 
{
    /**
     * Utility class should never be instantiated.
     */
    private GraphicsUtil() 
    {
    }

    /**
     * Saves image object as jpeg file.
     * <p>
     *
     * Thanks to Marco Schmidt for his example code.
     *
     * @deprecated   #saveImageAsJPEG
     */
    static public void saveImageAsJpeg( String filename
                                        , Image image )
        throws IOException
    {
        saveImageAsJPEG( filename, image );
    }

    /**
     * Saves image object as jpeg file.
     * <p>
     *
     * Thanks to Marco Schmidt for his example code.
     */
    static public void saveImageAsJPEG( String filename
                                        , Image image )
        throws IOException
    {
        if ( image instanceof BufferedImage )
        {
            saveImageAsJPEG( filename, (BufferedImage) image );
        }
        else
        {
            BufferedImage bufferedImage = new BufferedImage
                ( image.getWidth( null )
                  , image.getHeight( null )
                  , BufferedImage.TYPE_INT_RGB );
            Graphics2D graphics2D = bufferedImage.createGraphics();
            graphics2D.drawImage( image
                                  , 0
                                  , 0
                                  , image.getWidth( null )
                                  , image.getHeight( null )
                                  , null );
            
            saveImageAsJPEG( filename, bufferedImage );
            bufferedImage.flush();
        }
    }

    /**
     * Saves image object as jpeg file.
     * <p>
     *
     * Thanks to Marco Schmidt for his example code.
     *
     * @deprecated   #saveImageAsJPEG
     */
    static public void saveImageAsJpeg( String filename 
                                        , BufferedImage bufferedImage )
        throws IOException
    {
        saveImageAsJPEG( filename, bufferedImage );
    }

    /**
     * Saves image object as jpeg file.
     * <p>
     *
     * Thanks to Marco Schmidt for his example code.
     */
    static public void saveImageAsJPEG( String filename 
                                        , BufferedImage bufferedImage )
        throws IOException
    {
        BufferedOutputStream out = new BufferedOutputStream
               ( new FileOutputStream( FileUtil.normalizeFileName( filename ) ) );
        JPEGImageEncoder encoder = JPEGCodec.createJPEGEncoder( out );
        JPEGEncodeParam param = encoder.
               getDefaultJPEGEncodeParam( bufferedImage );
        float quality = 100.0f;
        param.setQuality( quality / 100.0f, false );
        encoder.setJPEGEncodeParam( param );
        encoder.encode( bufferedImage );

        out.close();
    }

    /**
     * Creates an image of the component and saves it in jpeg format.
     * <p>
     *
     * Taken from Lawrence H. Rodrigues: Building Imaging Applications with Java Technology.
     *
     * @deprecated   #saveComponentAsJPEG
     */
    static public void saveComponentAsJpeg( String filename
                                            , Component component 
                                            , Container container )
        throws IOException
    {
        saveComponentAsJPEG( filename, component, container );
    }

    /**
     * Creates an image of the component and saves it in jpeg format.
     * <p>
     *
     * Taken from Lawrence H. Rodrigues: Building Imaging Applications with Java Technology.
     */
    static public void saveComponentAsJPEG( String filename
                                            , Component component 
                                            , Container container )
        throws IOException
    {
        BufferedImage bufferedImage = componentToImage( component, container );
        saveImageAsJpeg( filename, bufferedImage );
        bufferedImage.flush();
    }

    /**
     * Creates an image of the component.
     *
     * @param    container   e.g. jframe.getContentPane().
     *
     * @author   Taken from Lawrence H. Rodrigues: Building Imaging Applications with Java Technology.
     */
    static public BufferedImage componentToImage( Component component 
                                                  , Container container )
        throws IOException
    {
        Rectangle recComponent = component.getBounds();
        BufferedImage bufferedImage = new BufferedImage
               ( recComponent.width
                 , recComponent.height
                 , BufferedImage.TYPE_INT_RGB );
        Graphics2D graphics2D = bufferedImage.createGraphics();

        SwingUtilities.paintComponent( graphics2D
                                       , component
                                       , container
                                       , 0
                                       , 0
                                       , recComponent.width
                                       , recComponent.height );

        return bufferedImage;
    }

    /**
     * Creates a new buffer image by cropping out a part of the
     * given image.
     */
    static public BufferedImage crop( Image image, int x, int y, int width, int height )
    {
        BufferedImage bufferedImage = null;

        if ( !( image instanceof BufferedImage ) )
        {
            bufferedImage = createBufferedImage( image );
        }
        else
        {
            bufferedImage = (BufferedImage) image;
        }

        return crop( bufferedImage, x, y, width, height );
    }

    /**
     * Creates a new buffer image by cropping out a part of the
     * given image.
     */
    static public BufferedImage crop( BufferedImage bufferedImage, int x, int y, int width, int height )
    {
        BufferedImage retImage = new BufferedImage( width
                                                    , height
                                                    , BufferedImage.TYPE_INT_RGB );
        Graphics2D g = retImage.createGraphics();
        AffineTransform t = AffineTransform.getTranslateInstance( - x, - y );
        g.transform( t );
        g.drawImage( bufferedImage, null, 0, 0 );

        return retImage;
    }

    /**
     * Saves a buffered image as a gif file.
     *
     * @param   filename   the file name will be normalized, which means the 
     *                     current user dir will be taken into account if 
     *                     necessary.
     *
     * @throws   IOException   if writing the file fails.
     */
    static public void saveImageAsGIF( String filename
                                       , BufferedImage image )
        throws IOException
    {
        Gif89Encoder gifenc = new Gif89Encoder( image );
        gifenc.setTransparentIndex( -1 );
        gifenc.getFrameAt( 0 ).setInterlaced( false );

        BufferedOutputStream out = new BufferedOutputStream
            ( new FileOutputStream( FileUtil.normalizeFileName( filename ) ) );
        gifenc.encode(out);
        out.close();
    }

    /**
     * Saves image object as gif file.
     *
     * @param   filename   the file name will be normalized, which means the 
     *                     current user dir will be taken into account if 
     *                     necessary.
     *
     * Thanks to Marco Schmidt for his example code.
     */
    static public void saveImageAsGIF( String filename
                                        , Image image )
        throws IOException
    {
        if ( image instanceof BufferedImage )
        {
            saveImageAsGIF( filename, (BufferedImage) image );
        }
        else
        {
            BufferedImage bufferedImage = new BufferedImage
                ( image.getWidth( null )
                  , image.getHeight( null )
                  , BufferedImage.TYPE_INT_RGB );
            Graphics2D graphics2D = bufferedImage.createGraphics();
            graphics2D.drawImage( image
                                  , 0
                                  , 0
                                  , image.getWidth( null )
                                  , image.getHeight( null )
                                  , null );
            
            saveImageAsGIF( filename, bufferedImage );
            bufferedImage.flush();
        }
    }

    /**
     * Creates a buffered image out of the given image.
     *
     * @param   filename   the file name will be normalized, which means the 
     *                     current user dir will be taken into account if 
     *                     necessary.
     *
     * @return   returns always a new buffered image.
     */
    static public BufferedImage createBufferedImage( Image image )
        //throws IOException
    {
        BufferedImage bufferedImage = new BufferedImage
            ( image.getWidth( null )
              , image.getHeight( null )
              , BufferedImage.TYPE_INT_RGB );
        Graphics2D graphics2D = bufferedImage.createGraphics();
        graphics2D.drawImage( image
                              , 0
                              , 0
                              , image.getWidth( null )
                              , image.getHeight( null )
                              , null );
        /*bufferedImage.flush();*/

        return bufferedImage;
    }

    /**
     * Compares two images pixel by pixel. At the current state,
     * no special consideration for transparency is taken into
     * account.
     *
     * @return   true   if all pixels equal each other in both
     *                  images.
     */
    static public boolean equals( ImageIcon iconA, ImageIcon iconB )
    {
        if ( iconA == null && iconB == null )
        {
            return true;
        }
        if ( iconA == null || iconB == null )
        {
            return false;
        }
        if ( iconA.getIconWidth() != iconB.getIconWidth()
             || iconA.getIconHeight() != iconB.getIconHeight() )
        {
            return false;
        }

        return equals( iconA.getImage(), iconB.getImage() );
    }

    /**
     * Compares two images pixel by pixel. At the current state,
     * no special consideration for transparency is taken into
     * account.
     *
     * @return   true   if all pixels equal each other in both
     *                  images.
     */
    static public boolean equals( Image imageA, Image imageB )
    {
        if ( imageA == null && imageB == null )
        {
            return true;
        }
        if ( imageA == null || imageB == null )
        {
            return false;
        }
        if ( imageA.getWidth( null ) != imageB.getWidth( null )
             || imageA.getHeight( null ) != imageB.getHeight( null ) )
        {
            return false;
        }

        BufferedImage bufferedImageA = null;
        BufferedImage bufferedImageB = null;
        if ( imageA instanceof BufferedImage )
        {
            bufferedImageA = (BufferedImage) imageA;
        }
        else
        {
            bufferedImageA = createBufferedImage( imageA );
        }
        if ( imageB instanceof BufferedImage )
        {
            bufferedImageB = (BufferedImage) imageB;
        }
        else
        {
            bufferedImageB = createBufferedImage( imageB );
        }

        return equals( bufferedImageA, bufferedImageB );
    }

    /**
     * Compares two images pixel by pixel. At the current state,
     * no special consideration for transparency is taken into
     * account. Be aware that transient images might show black
     * pixels where you don't expect them.
     *
     * @return   true   if all pixels equal each other in both
     *                  images.
     */
    static public boolean equals( BufferedImage imageA, BufferedImage imageB )
    {
        Util.debug( GraphicsUtil.class, "equals(..).ENTER" );
        if ( imageA == null && imageB == null )
        {
            return true;
        }
        if ( imageA == null || imageB == null )
        {
            return false;
        }
        if ( imageA.getWidth( null ) != imageB.getWidth( null )
             || imageA.getHeight( null ) != imageB.getHeight( null ) )
        {
            return false;
        }

        Raster rasterA = imageA.getRaster();
        Raster rasterB = imageB.getRaster();
        SampleModel sampleModelA = rasterA.getSampleModel();
        SampleModel sampleModelB = rasterB.getSampleModel();
        DataBuffer dataBufferA = rasterA.getDataBuffer();
        DataBuffer dataBufferB = rasterB.getDataBuffer();
        
        int imageWidth  = imageA.getWidth( null );
        int imageHeight = imageA.getHeight( null );
        for( int y = 0; y < imageHeight; y ++ )
        {
            for( int x = 0; x < imageWidth; x ++ )
            {
                int pix[] = null;
                int[] pixelA = sampleModelA.getPixel( x, y, pix, dataBufferA );
                int[] pixelB = sampleModelB.getPixel( x, y, pix, dataBufferB );
                if ( pixelA.length != pixelB.length )
                {
                    return false;
                }
                for( int pixel = 0; pixel < pixelA.length; pixel ++ )
                {
                    Util.debug( GraphicsUtil.class, "equals(..).pixelA[" + x + "," + y + "," + pixel + "]: " 
                                + pixelA[ pixel ] );
                    Util.debug( GraphicsUtil.class, "equals(..).pixelB[" + x + "," + y + "," + pixel + "]: " 
                                + pixelB[ pixel ] );
                    if ( pixelA[ pixel ] != pixelB[ pixel ] )
                    {
                        return false;
                    }
                }
            }
        }
                
        Util.debug( GraphicsUtil.class, "equals(..).EXIT" );

        return true;
    }
}
