/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the reusable ccl java library
 * (http://www.kclee.com/clemens/java/ccl/).
 *
 * The Initial Developer of the Original Code is
 * Chr. Clemens Lee.
 * Portions created by Chr. Clemens Lee are Copyright (C) 2002
 * Chr. Clemens Lee. All Rights Reserved.
 *
 * Contributor(s): Chr. Clemens Lee <clemens@kclee.com>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

package ccl.swing;

import ccl.util.FileObject;
import ccl.util.Init;
import ccl.util.Util;
import java.awt.Event;
import java.awt.event.KeyEvent;
import java.util.Enumeration;
import java.util.Vector;
import javax.help.CSH;
import javax.help.HelpBroker;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.KeyStroke;

/**
 * A convenience class to create application menus in an easy way.
 * This class is mainly used by objects of class MainJFrame.
 * But sometimes you might need to look heere to understand how
 * the menus got created and how they are structured.
 *
 * // ToDo: Info und Hilfe standardmig bearbeiten
 * Shortcut ctrl-s wird automatisch fr save oder speichern eingefgt
 * in MainMenu.MainMenu().
 *
 * @author  Chr. Clemens Lee
 * @version $Id: MainJMenuBar.java,v 1.21 2003/05/01 16:44:22 clemens Exp clemens $
 */
public class MainJMenuBar extends JMenuBar 
{
    public static final String S_HELP_CONTENTS = "&Contents...";

    private MainJFrame _pMainJFrame = null;

    /**
     * Creates a custom menu bar object which gets its menu item
     * descriptions from a collection of name strings or runnable
     * action objects. The name of the menu bar entry is the first
     * object, a string, in the sub vector.
     */
    public MainJMenuBar(Vector vMenus_, MainJFrame pMainJFrame_) 
    {
        super();

        Util.panicIf(vMenus_ == null || pMainJFrame_ == null);
        Util.debug("MainJMenuBar: init");
        
        _pMainJFrame = pMainJFrame_;
        
        for(Enumeration e = vMenus_.elements(); e.hasMoreElements(); ) 
        {
            Vector vMenu = (Vector)e.nextElement();
            MainJMenu pMainJMenu = new MainJMenu(vMenu, pMainJFrame_);
            Util.debug("MainJMenuBar: pMainMenu: " + pMainJMenu);
            add(pMainJMenu);
        }
    }

    /**
     * In the file menu above the exit item is a section with
     * old projects which has been opened before.
     * This section is separated by menu separators. If there has
     * not been any previous project open, only one separator
     * exists above the exit item.
     * 
     * This method updates the list with old projects (files).
     * It removes the old list, and uses the init object to
     * fill the new list.
     *
     * This method is thread safe but might not be executed immediately.
     */
    public synchronized void updateMenu() 
    {
        SwingUtil.invokeLaterIfNecessary( new Runnable() {
            public void run() {
                MainJMenu pFileJMenu = (MainJMenu)getMenu(0);
                Util.debug( this, "updateMenu().pFileJMenu: " + pFileJMenu);
                pFileJMenu.removeOldFileItems();
                pFileJMenu.insertOldFiles(_pMainJFrame);
            }
        } );
    }

    class MainJMenu extends JMenu 
    {
        /** First index of old files (projects) presented in file menu. */
        private int _oldFilesStartIndex = -1;
        /** Number of old files (projects) presented in file menu. */
        private int _oldFilesCount      =  0;

        /**
         * Creates a vector with strings representing menu item names
         * which are equal to the file names of the most recent
         * project files.<p>
         *
         * If the returned vector is not empty, its last element
         * is a string representing a separator menu item.
         * The following entries will start with a number  
         * (beginning with one and then increasing) followed by 
         * a space before the base file name with the first char 
         * always in upper case follows.
         */
        private Vector _getOldFileItems(Init pInit_) 
        {
            Vector vRetVal = new Vector();
        
            Enumeration eOldFiles = pInit_.getOldFilesElements();
            if (eOldFiles.hasMoreElements()) 
            {
                vRetVal.addElement("--------");
            }
        
            int nr = pInit_.getOldFilesSize();
            _oldFilesCount = nr;
            for( ; eOldFiles.hasMoreElements(); ) 
            {
                vRetVal.insertElementAt("" + nr + " " +
                                        Util.firstCharToUpperCase
                                        (((FileObject)eOldFiles.
                                          nextElement())
                                         .getName()),
                                        0);
                nr--;
            }
            
            return vRetVal;
        }
        
        private Enumeration _getMenuBodyElements(Vector vMenu_, Init pInit_) 
        {
            Enumeration eRetVal = null;
            String sLabel = (String)vMenu_.elementAt(0);
            
            if (sLabel.equals("Datei") || sLabel.equals("File") ||
                sLabel.equals("&Datei") || sLabel.equals("&File"))
            {
                Vector vFileMenu = (Vector)vMenu_.clone();
                
                // --- Alte Dateien in Menu einfgen ---
                
                int menuSlot = vFileMenu.size() - 1;
                if (menuSlot > 1) 
                {
                    vFileMenu.insertElementAt("--------", menuSlot);
                    menuSlot++;
                }
                _oldFilesStartIndex = menuSlot-1;
                
                Vector vOldFileItems = _getOldFileItems(pInit_);
                
                vFileMenu = Util.insert(vFileMenu, vOldFileItems, menuSlot);
                
                eRetVal = vFileMenu.elements();
            }
            else 
            {
                eRetVal = vMenu_.elements();
            }
            eRetVal.nextElement();
            
            return eRetVal;
        }

        public MainJMenu(Vector vMenu_, MainJFrame pMainJFrame_) 
        {
            super((String)vMenu_.elementAt(0));
            
            setName( getText() );
            
            if ( getText().equals( "File" ) ) 
            {
                setMnemonic( 'F' );
            }
            if ( getText().equals( "Edit" ) ) 
            {
                setMnemonic( 'E' );
            }
            if ( getText().equals( "Help" ) ) 
            {
                setMnemonic( 'H' );
            }
            
            Util.panicIf(vMenu_ == null || vMenu_.size() < 2);
            Util.panicIf(pMainJFrame_ == null);
            
            for(Enumeration e = _getMenuBodyElements(vMenu_,
                                                     pMainJFrame_.getInit());
                e.hasMoreElements(); )
            {
                Object oNext = e.nextElement();
                
                if (oNext instanceof Vector) 
                {
                    // Rekursiv unter Menues
                    JMenu mSub = new MainJMenu((Vector)oNext,
                                               pMainJFrame_);
                    add(mSub);
                }
                else 
                {
                    if ( oNext instanceof RunnableAction ) 
                    {
                        RunnableAction pRunnableAction = (RunnableAction)oNext;
                        pRunnableAction.setMainFrame
                               ( pMainJFrame_ );

                        JMenuItem miTemp = add( pRunnableAction );
                        Character charMnemonic = 
                               pRunnableAction.getMnemonic();
                        if ( charMnemonic != null ) 
                        {
                            miTemp.setMnemonic( charMnemonic.charValue() );
                        }
                        
                        KeyStroke ksAccelerator = pRunnableAction.getAccelerator();
                        if ( ksAccelerator != null ) 
                        {
                            miTemp.setAccelerator( ksAccelerator );
                        }
                        miTemp.setName( miTemp.getText() );
                        Util.debug( this, "<init>.miTemp.name: " +
                                    miTemp.getName() );
                    }
                    else 
                    {
                        String sActionCommand = (String)oNext;
                        String sMenuItem = sActionCommand;
                        
                        int indexMnemonic = sMenuItem.indexOf( '&' );
                        char cMnemonic = '&';
                        if ( indexMnemonic != -1 && 
                             indexMnemonic < sMenuItem.length() ) 
                        {
                            cMnemonic = sMenuItem.charAt( indexMnemonic + 1 );
                            sMenuItem = Util.replace( sMenuItem, "&", "" );
                        }
                        Util.debug(sMenuItem);
                        if (sMenuItem.charAt(0) == '-') 
                        {
                            Util.debug("---");
                            addSeparator();
                        }
                        else 
                        {
                            JMenuItem miTemp = null;
                            if (sMenuItem.equals("Save") || sMenuItem.equals("Speichern")) 
                            {
                                miTemp = new JMenuItem( sMenuItem, 'S' );
				miTemp.setAccelerator( KeyStroke.getKeyStroke( KeyEvent.VK_S, Event.CTRL_MASK ) );
                                /*miTemp.setShortcut(new MenuShortcut(KeyEvent.VK_S));
				KeyStroke ksCtrlS = KeyStroke.getKeyStroke('c', Event.CTRL_MASK);
                                  miTemp.registerKeyboardAction(pMainJFrame_*/
                            }
                            else 
                            {
                                miTemp = new JMenuItem(sMenuItem);
                            }
                            if ( indexMnemonic != '&' ) 
                            {
                                miTemp.setMnemonic( cMnemonic );
                            }
                            miTemp.setActionCommand( sActionCommand );
                            miTemp.setName( miTemp.getText() );
                            // add help action listener?
                            if ( sActionCommand.equals( MainJMenuBar.S_HELP_CONTENTS ) ) 
                            {
                                Util.debug( this, "pMainJFrame_: " + pMainJFrame_ );
                                HelpBroker pHelpBroker = 
                                       pMainJFrame_.getHelpBroker();
                                miTemp.addActionListener
                                       ( new CSH.
                                         DisplayHelpFromSource
                                         ( pHelpBroker ) );
                                Util.debug( this, "<init>(..).HELP BROKER IS SET" );
                            }
                            else 
                            {
                                miTemp.addActionListener(pMainJFrame_);
                            }
                            add(miTemp);
                        }
                    }
                }
            }
        }
        
        /**
         * Remove list in file menu of old files (projects).
         */
        public void removeOldFileItems() 
        {
            if (_oldFilesCount == 0) 
            {
                return;
            }
            for(int i = 0; i < _oldFilesCount + 1; i++) 
            {
                remove(_oldFilesStartIndex);
            }
            _oldFilesCount = 0;
        }
        
        public void insertOldFiles(MainJFrame pMainJFrame_) 
        {
            Vector vOldFileItems = _getOldFileItems(pMainJFrame_.getInit());
            
            for(int i = 0; i < vOldFileItems.size(); i ++)
            {
                String sItem = (String)vOldFileItems.elementAt(i);
                if (sItem.charAt(0) == '-') 
                {
                    insertSeparator(_oldFilesStartIndex + i);
                }
                else 
                {
                    JMenuItem miTemp = new JMenuItem(sItem);
                    miTemp.setMnemonic( sItem.charAt( 0 ) );
                    miTemp.addActionListener(pMainJFrame_);
                    insert(miTemp, _oldFilesStartIndex + i);
                }
            }
        }
    }
}
