
//-----------------------------------------------------------------------------
// Some helper functions used by more or less every DirectX application
//
// Copyright (c) Microsoft Corporation 2005-2006.
// This sample code is provided "as is" without warranty of any kind. 
// We disclaim all warranties, either express or implied, including the 
// warranties of merchantability and fitness for a particular purpose. 
//-----------------------------------------------------------------------------

#light

module DirectXLib

open Compatibility
open System
open System.Drawing
open System.Windows.Forms
open Microsoft.DirectX
open Microsoft.DirectX.Direct3D

//-----------------------------------------------------------------------------
// Float32 arithmetic is standard
//-----------------------------------------------------------------------------

let PI = float32(Math.PI)

//-----------------------------------------------------------------------------
// Create a stream of Float32 time values
//-----------------------------------------------------------------------------

let time = 
   let startTime = Environment.TickCount in 
   (fun () -> float32(Environment.TickCount - startTime))
   
//-----------------------------------------------------------------------------
// A standard function to create a standard form
//-----------------------------------------------------------------------------

let NewStandardForm(title,width,height) : Form = 
    new Form(Text = "F# Direct3D Tutorial 6 - Meshes",
             ClientSize = new Size(400, 300),
             Visible = true)

//-----------------------------------------------------------------------------
// A standard function to create a standard device
//-----------------------------------------------------------------------------

let NewStandardDevice(form : Form) = 
  let presentParams = new PresentParameters(Windowed = true,
                                            SwapEffect = SwapEffect.Discard,
                                            // Turn on a Depth stencil
                                            EnableAutoDepthStencil = true,
                                            // And the stencil format
                                            AutoDepthStencilFormat = DepthFormat.D16)
  new Device(0, DeviceType.Hardware, form,
             CreateFlags.SoftwareVertexProcessing, 
             Idioms.ParamArray [presentParams]) 

//-----------------------------------------------------------------------------
// Abstract away some of the gore from DirectX programming.  For these samples
// we simply want to render some stuff on a form.  
//-----------------------------------------------------------------------------

type Renderer = 
  { OnDeviceReset : Device -> RendererForDevice }
and RendererForDevice = 
  { Render: unit -> unit }

let SimpleRenderer (f : Device -> unit) = 
  { OnDeviceReset = (fun device -> { Render = fun () -> f device} ) }

let Renderer f = { OnDeviceReset = f }
let RendererForDevice f = { Render = f }

let RenderInLoop((form:Form), (device:Device), renderers) =
  // We reapply the renderers to the device upon each reset
  
  let actions = ref [] in 
  let OnDeviceReset () =
    // Turn on the zbuffer and turn on ambient lighting 
    device.RenderState.ZBufferEnable <- true;
    device.RenderState.Ambient <- System.Drawing.Color.White;
    actions :=  List.map (fun f -> f.OnDeviceReset(device)) renderers; in

  let invisible = ref false in 
  let CheckForMinimized() =
        invisible := (form.WindowState = FormWindowState.Minimized) 
                 || (form.Visible = false) in 

  let render() = 
    if not !invisible then begin
      device.BeginScene();
      List.iter (fun f -> f.Render()) !actions;
      device.EndScene();
      device.Present()
    end in
    
  form.Resize.Add(fun _ -> CheckForMinimized())
  form.Activated.Add(fun _ -> CheckForMinimized());
  form.Paint.Add(fun _ -> render());
  device.DeviceReset.Add(fun _ -> OnDeviceReset());
  OnDeviceReset();
  while form.Created do
        render();
        Application.DoEvents();
  done

//-----------------------------------------------------------------------------
// A little library of standard rendering functions
//-----------------------------------------------------------------------------

let ChangingWorldTransform(worldf) = SimpleRenderer (fun device -> device.Transform.World <- worldf())
let Clear(color:Color) = SimpleRenderer (fun device -> device.Clear((ClearFlags.ZBuffer ||| ClearFlags.Target), color, 1.0f, 0))
let View(view) = SimpleRenderer (fun device -> device.Transform.View <- view)
let Projection(proj) = SimpleRenderer (fun device -> device.Transform.Projection <- proj)
  
//-----------------------------------------------------------------------------
// A renderer for colored points
//-----------------------------------------------------------------------------

let RenderColoredPointTriangles(triangles) =
 Renderer(fun device ->   
  let n = Array.length triangles  in 
  let vertexBuffer = 
    new VertexBuffer(typeof<CustomVertex.PositionColored>,
                     3 * n, device, Usage.None, CustomVertex.PositionColored.Format, 
                     Pool.Default) in
  let verts = 
     CompatArray.init (n * 3) (fun i -> 
       let p1,p2,p3 = triangles.(i / 3) in 
       let (a,b,c,(d:Color)) = match i % 3 with | 0 -> p1 | 1 -> p2 | _ -> p3 in
       new CustomVertex.PositionColored(a,b,c,d.ToArgb()))  in 
  let FillVertexBuffer() =
    let stm = vertexBuffer.Lock(0, 0, LockFlags.None) in
    stm.Write(verts); 
    vertexBuffer.Unlock() in 
  vertexBuffer.Created(fun _ -> FillVertexBuffer());
  FillVertexBuffer();
  RendererForDevice(fun () ->
    device.SetStreamSource( 0, vertexBuffer, 0);
    device.VertexFormat <- CustomVertex.PositionColored.Format;
    device.DrawPrimitives(PrimitiveType.TriangleList, 0, n)))
