
//----------------------------------------------------------------------------
// On non-Windows platforms this file must be compiled with FEATURE_PAL
//
// On Windows platforms this file can compile with or without FEATURE_PAL.  
// If compiled _with_ FEATURE_PAL then the functions will only work 
// when run against an installation of the SSCLI.  If compiled without FEATURE_PAL
// then on Windows platforms the functions will work both against a Microsoft CLR
// and against an SSCLI installation.

#include "common.h"

#include <ole2.h>
#include "cordefs.h"
#include "getcor.h"


HINSTANCE  g_RtDll = NULL; // Handle to mscoree.dll or sscoree.dll

// Fetch a handle to either a .NET or SSCLI Common Language Runtime.  The handle is used
// to find the directory where assemblies like mscorlib and System.dll live, so we can
// read their metadata.  
//
// We use dynamic linking to make sure we fail late rather than early, i.e. so the compiler is
// still usable for some functions even without a CLR installed, and that it is usable
// even if you only have a SSCLI CLR installed (and no Microsoft CLR), or only a 
// Microsoft CLR (and no SSCLI CLR).
HRESULT GetDllForEE(bool forceUseOfSSCLI)
{
  WCHAR buf[_MAX_PATH + 1];
  ULONG32 len;
  char res[_MAX_PATH + 1];
  
  if (!g_RtDll)
  {
      // On Windows platforms look for the SSCLI DLL if ordered to, which 
      // makes AbsIL tools usable against both an SSCLI and a Microsoft CLR
      WCHAR *dll = NULL;
#if !(defined FEATURE_PAL) && !(defined PLATFORM_UNIX)
      if (forceUseOfSSCLI)  
          dll = L"sscoree.dll";
      else
          dll = MSCOREE_SHIM_W;
#else
      dll = MSCOREE_SHIM_W;
#endif
      g_RtDll = LoadLibraryW(dll);
      if (!g_RtDll)
      {
          fprintf(stderr, "%S could not be found\n", dll);
          return E_FAIL;
      }
  }
  return S_OK;
}

//----------------------------------------------------------------------------
//  CORSystemDirectory
//
//----------------------------------------------------------------------------

#define EXTAPI extern "C" HRESULT  _declspec(dllexport)

typedef HRESULT __stdcall CorBindToRuntimeExProc(LPWSTR,LPWSTR,DWORD,REFCLSID,REFIID,LPVOID*);

EXTAPI PInvoke_CORSystemDirectory(bool forceUseOfSSCLI, LPTSTR res, ULONG32 bufSize)
{

  GetCORSystemDirectoryProc *dllGetCorSysDir;
  
  HRESULT hr = GetDllForEE(forceUseOfSSCLI);
  if (FAILED(hr)) return hr;
  if (dllGetCorSysDir = (GetCORSystemDirectoryProc *) GetProcAddress(g_RtDll, "GetCORSystemDirectory"))
  {
      ULONG32 len;
      if ( SUCCEEDED((*dllGetCorSysDir)(res, bufSize, &len)))
      {
	       return S_OK;
      }
  }

  return E_FAIL;
}  

#ifdef CAML_STUBS
extern "C"
CAMLprim value CORSystemDirectoryCaml(value forceUseOfSSCLI)
{
  CAMLparam1(forceUseOfSSCLI);

  WCHAR buf[_MAX_PATH + 1];
  
  HRESULT hr = PInvoke_CORSystemDirectory(Bool_val(forceUseOfSSCLI), buf, _MAX_PATH);

  if (FAILED(hr))
	  failwith("*** A call to GetCORSystemDirectory failed.  You need to either\n*** repair your installation of the CLR or set the variables\n*** 'COMPLUS_InstallRoot' and 'COMPLUS_Version'.");

  char res[_MAX_PATH + 1];
  WideCharToMultiByte(CP_ACP, 0, buf, -1, res, _MAX_PATH, NULL, NULL);

  CAMLreturn(copy_string(res));
      
}

#endif // CAML_STUBS


//----------------------------------------------------------------------------
//  BindToCor


EXTAPI pCorBindToRuntimeExProc(WCHAR *pVersion, bool forceUseOfSSCLI)
{
    ICorRuntimeHost *pHost;
    
    CorBindToRuntimeExProc *pCorBindToRuntimeExProc;
    HRESULT hr = S_OK;
    if (SUCCEEDED(hr = GetDllForEE(forceUseOfSSCLI)))
    {
        //fprintf(stderr,"finding entry point...\n"); fflush(stderr);
        if (pCorBindToRuntimeExProc = (CorBindToRuntimeExProc *) GetProcAddress(g_RtDll, "CorBindToRuntimeEx"))
        {
            //fprintf(stderr,"binding to runtime...\n"); fflush(stderr);
            hr = (*pCorBindToRuntimeExProc)(pVersion,NULL,0,CLSID_CorRuntimeHost,IID_ICorRuntimeHost,(void **) &pHost);
            //fprintf(stderr,"hr2 = 0x%x...\n", hr); fflush(stderr);
        }
    }
    return hr;
}

//----------------------------------------------------------------------------
//  absilWriteGetMetadataVersion
//
//----------------------------------------------------------------------------

typedef HRESULT __stdcall GetCORVersionProc(WCHAR *, ULONG32, ULONG32 *);

EXTAPI absilWriteGetMetadataVersion(bool forceUseOfSSCLI, LPTSTR buf, ULONG32 bufSize)
{
  GetCORVersionProc *dllGetCorVersion;

  HRESULT hr = GetDllForEE(forceUseOfSSCLI);
  if (FAILED(hr)) return hr;

  if (dllGetCorVersion = (GetCORVersionProc *) GetProcAddress(g_RtDll, "GetCORVersion"))
  {
      ULONG32 len;
      if ( SUCCEEDED((*dllGetCorVersion)(buf, _MAX_PATH, &len)))
      {
          return S_OK;
      }
  }
  return E_FAIL;
   
}  

#ifdef CAML_STUBS
extern "C"
CAMLprim value absilWriteGetMetadataVersionCaml(value forceUseOfSSCLI)
{
  CAMLparam1(forceUseOfSSCLI);
  WCHAR buf[_MAX_PATH + 1];
  HRESULT hr = absilWriteGetMetadataVersion(Bool_val(forceUseOfSSCLI), buf, _MAX_PATH);
  if (FAILED(hr))
     failwith("*** A call to GetCORVersion failed.  You need to either\n*** repair your installation of the CLR or set the variables\n*** 'COMPLUS_InstallRoot' and 'COMPLUS_Version'.");
          
  char res[_MAX_PATH + 1];
  WideCharToMultiByte(CP_ACP, 0, buf, -1, res, _MAX_PATH, NULL, NULL);

  CAMLreturn(copy_string(res));
}

extern "C"
CAMLprim value pCorBindToRuntimeExProcCaml(value version, value forceUseOfSSCLI)
{
  CAMLparam1(forceUseOfSSCLI);
  WCHAR versionw[_MAX_PATH];
  MultiByteToWideChar(CP_ACP, 0, String_val(version), -1, versionw, _MAX_PATH);

  HRESULT hr = pCorBindToRuntimeExProc(versionw, Bool_val(forceUseOfSSCLI));
  if (FAILED(hr))
     failwith("Could not bind to the given CLR version");
  CAMLreturn(Atom(0));
}


#endif // CAML_STUBS

//----------------------------------------------------------------------------
//  GetModuleFileName
//
//----------------------------------------------------------------------------


#ifdef CAML_STUBS
extern "C"
CAMLprim value GetModuleFileNameCaml()
{
    CAMLparam0();
    WCHAR wszFsPath[_MAX_PATH]; // compiler path
    DWORD dwLen = ::GetModuleFileNameW(NULL, wszFsPath, _MAX_PATH);
    char res[_MAX_PATH + 1];
    WideCharToMultiByte(CP_ACP, 0, wszFsPath, -1, res, _MAX_PATH, NULL, NULL);
    if (dwLen == 0)
        failwith("GetModuleFileName failed");
    if (dwLen == MAX_PATH)
        failwith("GetModuleFileName: insufficient bufer");
    CAMLreturn(copy_string(res));
}


#endif // CAML_STUBS




