(* (c) Microsoft Corporation. All rights reserved *)

(*F#
/// A set of "IL rewrites" ("morphs").  These map each sub-construct
/// of particular types.  The morphing functions are passed
/// some details about the context in which the item being
/// morphed occurs, e.g. the module being morphed itself, the
/// type_def (possibly nested) where the item occurs, 
/// the method_def (if any) where the item occurs. etc.
module Microsoft.Research.AbstractIL.Morphs 
open Microsoft.Research.AbstractIL 
open Microsoft.Research.AbstractIL.Internal 
module Ilx = Microsoft.Research.AbstractIL.Extensions.ILX.Types 
module Il = Microsoft.Research.AbstractIL.IL 
F#*)  

open Il

type 'a morph = 'a -> 'a

(** A type representing a choice between two types A or B *)
type ('a, 'b) choice = Choice1of2 of 'a | Choice2of2 of 'b

(** A type representing a choice between three types A, B or C *)
type ('a, 'b, 'c) choice3 =
  | Choice1of3 of 'a
  | Choice2of3 of 'b
  | Choice3of3 of 'c

(* -------------------------------------------------------------------- 
 * Morphs over code
 * -------------------------------------------------------------------- *)

(** Morph all instructions in a basic block *)
val bblock_instr2instr: instr morph -> basic_block -> basic_block

(*
/// Morph all instructions in a basic block, converting each to a non-branching 
/// sequence of instructions.
*)
val bblock_instr2instrs: (instr -> instr list) -> basic_block -> basic_block

(*
/// Morph each instruction in a basic block to either an instruction or a block.  If morphed to
/// a block then the block should use the given enter/exit labels.
*)
val bblock2code_instr2code: (code_label -> code_label -> instr -> (instr list, code) choice) -> basic_block -> code

(** Morph each basic block in a code object to a structured code tree, and simultaneously morph types. *)
val block_bblock2code_typ2typ: (basic_block -> code) * (typ morph) -> code -> code

(** Morph each basic block in a code object to a structured code tree, and simultaneously morph types. *)
val code_bblock2code_typ2typ: (basic_block -> code) * typ morph -> code -> code

val topcode_bblock2code_typ2typ: (basic_block -> code) * typ morph -> code -> code

(** Morph each basic block in a code object to a structured code tree *)
val block_bblock2code: (basic_block -> code) -> code -> code

(** Morph each basic block in a code object to a structured code tree *)
val code_bblock2code: (basic_block -> code) -> code -> code

(** Morph each basic block in a code object to a structured code tree *)
val topcode_bblock2code: (basic_block -> code) -> code -> code

(** Morph the code of a method definition *)
val mdef_code2code: code morph  -> method_def -> method_def

(** Morph each scope reference inside a type signature *)
val tref_scoref2scoref: scope_ref morph -> type_ref -> type_ref 

(** Morph each scope reference inside a type signature *)
val typ_scoref2scoref: scope_ref morph -> typ -> typ 

(** Morph each type variable inside a syntactic type signature to a type *)
val typ_tyvar2typ: (Nums.u16 -> typ) -> typ -> typ

(** Morph each type reference inside a syntactic type signature *)
val typ_tref2tref: type_ref morph -> typ -> typ  

(* -------------------------------------------------------------------- 
 * Morph types in signatures, method references etc.  
 * -------------------------------------------------------------------- *)

(** Morph all types in a calling signature *)
val callsig_typ2typ: typ morph -> callsig -> callsig

(** Morph all types in genparams *)
val gparams_typ2typ: typ morph -> genparams -> genparams

(** Morph all types in a method reference *)
val mref_typ2typ: typ morph -> method_ref -> method_ref

(** Types occuring in the formal parts of signatures are given a context indicating which formal object they lie within. *)
type formal_scope_ctxt = (method_spec, field_spec, Ilx.classunion_spec) choice3

(** Morph all types in a method spec *)
val mspec_typ2typ: typ morph * (formal_scope_ctxt -> typ morph) -> method_spec -> method_spec

(** Morph all types in a field spec *)
val fspec_typ2typ: typ morph * (formal_scope_ctxt -> typ morph) -> field_spec -> field_spec

(** Morph the type in a local variable spec *)
val locals_typ2typ: typ morph -> local list -> local list

(** Morph the type in a method parameter spec *)
val param_typ2typ: typ morph -> param -> param

(** Morph all types in a list of method parameter specs *)
val params_typ2typ: typ morph -> param list -> param list

(** Morph all types in a set of custom attributes *)
val cattrs_typ2typ: typ morph -> custom_attrs -> custom_attrs

(** Morph the type in a method return value spec *)
val return_typ2typ: typ morph -> returnv -> returnv

(** Morph the type in an override specification *)
val ospec_typ2typ: typ morph -> overrides_spec -> overrides_spec

(** Morph types in instructions.  One morph for the actual types and another for the formal signature types *)
val instr_typ2typ: (instr option -> typ morph) * (instr option -> formal_scope_ctxt -> typ morph) -> instr -> instr

(** Morph all types in a field definition *)
val fdef_typ2typ: typ morph -> field_def -> field_def

(* -------------------------------------------------------------------- 
 * Morph tables 1:1
 * -------------------------------------------------------------------- *)

val mdefs_mdef2mdef: method_def morph -> methods -> methods
val fdefs_fdef2fdef: field_def morph -> fields -> fields
val tdefs_tdef2tdef: type_def morph -> types -> types (* nb. does not do nested tdefs *)
val fdefs_typ2typ: typ morph -> fields -> fields
val pdefs_typ2typ: typ morph -> properties -> properties
val edefs_typ2typ: typ morph -> events -> events
val mimpls_typ2typ: typ morph -> method_impls -> method_impls

(* -------------------------------------------------------------------- 
 * Morph tables 1:n
 * -------------------------------------------------------------------- *)

val mdefs_mdef2mdefs: (method_def -> method_def list) -> methods -> methods
val tdefs_tdef2tdefs: (type_def -> type_def list) -> types -> types

(* -------------------------------------------------------------------- 
 * Morph all tables of types in "Il.modul"
 * -------------------------------------------------------------------- *)

val module_tdefs2tdefs: types morph -> modul -> modul

(** Morph all scope (i.e. assembly/module) references throughout an entire module *)
val module_scoref2scoref:  scope_ref morph -> modul -> modul

(** Morph all type references throughout an entire module *)
val module_tref2tref:  type_ref morph ->  modul ->  modul

(** Morph all type references throughout an entire module. *)
val module_tref2tref_memoized:  type_ref morph ->  modul ->  modul

val module_scoref2scoref_memoized:  scope_ref morph ->  modul ->  modul

type enclosing_type_defs = type_def list * type_def

(** Morph all type signatures throughout an entire module *)
val module_typ2typ:
  (modul -> enclosing_type_defs option -> method_def option -> typ morph) ->
  modul -> modul

(** Morph all instructions throughout an entire module *)
val module_instr2instr:
  (modul -> enclosing_type_defs -> method_def option -> instr morph) ->
  modul -> modul

(*
/// Morph all type definitions throughout an entire module, including nested type 
/// definitions.
*)
val module_tdef2tdef: (type_def list -> type_def morph) -> modul -> modul

(*
/// Morph all method definitions throughout an entire module, including nested type 
/// definitions.
*)
val module_mdef2mdef: (enclosing_type_defs -> method_def morph) -> modul -> modul

(*
/// Morph all field definitions throughout an entire module, including nested type 
/// definitions.
*)
val module_fdef2fdef: (enclosing_type_defs -> field_def morph) -> modul -> modul

(*
/// Simultaneously morph all types and method bodies throughout a method definition.
/// The morphing functions should be consistent, e.g. the function that morphs method bodies should morph
/// types in the same way as the function that morphs types alone.  
*)
val mdef_typ2typ_ilmbody2ilmbody:
  (method_def option -> typ morph) *
  (method_def option -> il_method_body morph) ->
  method_def -> method_def

(*
/// Simultaneously morph all types and method bodies throughout a module.
/// The morphing functions should be consistent, e.g. the function that morphs method bodies should morph
/// types in the same way as the function that morphs types alone.  
*)
val module_typ2typ_ilmbody2ilmbody:
  (modul -> enclosing_type_defs option -> method_def option -> typ morph) *
  (* the next one is used to convert both closure and method bodies *)
  (modul -> enclosing_type_defs -> method_def option -> il_method_body morph) ->
  modul -> modul

val module_typ2typ_ilmbody2ilmbody_mdefs2mdefs:
  (modul -> enclosing_type_defs option -> method_def option -> typ morph) *
  (modul -> enclosing_type_defs -> method_def option -> il_method_body morph) *
  (modul -> enclosing_type_defs -> methods morph) ->
  modul -> modul

(*
/// Simultaneously morph all instructions and types throughout a module.
/// The morphing functions should be consistent, e.g. the function that morphs instructions should morph
/// types in the same way as the function that morphs types alone.  
*)
val module_instr2instr_typ2typ:
  (modul -> enclosing_type_defs -> method_def option -> instr morph) *
  (modul -> enclosing_type_defs option -> method_def option -> typ morph) ->
  modul -> modul

(* -------------------------------------------------------------------- 
 * Misc.
 * -------------------------------------------------------------------- *)

val module_instr2instrs_maxstack2maxstack:
  (modul -> type_def list * type_def -> method_def option -> instr -> instr list) *
  (modul -> type_def list * type_def -> method_def option -> Nums.i32 -> Nums.i32) ->
  modul -> modul

val mbody_ilmbody2ilmbody: il_method_body morph -> method_body -> method_body
val cloinfo_ilmbody2ilmbody: il_method_body morph -> Ilx.closure_info ->  Ilx.closure_info
val topcode_instr2instrs: (instr -> instr list) -> code -> code
val topcode_instr2code:
  (code_label -> code_label -> instr -> (instr list, code) choice) ->
  code -> code

(* -------------------------------------------------------------------- 
 * ILX related
 * -------------------------------------------------------------------- *)

val freevar_typ2typ: typ morph -> Ilx.freevar -> Ilx.freevar
val cuspec_typ2typ: typ morph * (formal_scope_ctxt -> typ morph) -> Ilx.classunion_spec -> Ilx.classunion_spec


