(* (c) Microsoft Corporation. All rights reserved *)
(*F# 
module Microsoft.Research.AbstractIL.Extensions.ILX.Nupp_erase
open Microsoft.Research.AbstractIL 
open Microsoft.Research.AbstractIL.Internal 
open Microsoft.Research.AbstractIL.Extensions.ILX
module Ilx = Microsoft.Research.AbstractIL.Extensions.ILX.Types 
module Ilmorph = Microsoft.Research.AbstractIL.Morphs  
module Ildiag = Microsoft.Research.AbstractIL.Diagnostics 
module Ilprint = Microsoft.Research.AbstractIL.AsciiWriter 
module Il = Microsoft.Research.AbstractIL.IL 
F#*)  

open Ildiag
open Nums
open Il
open Ilx
open Ilmorph
open Ilprint
open List

(* -------------------------------------------------------------------- 
 * Erase PP-at-representations by compiling down to code generation
 * demands.  
 *
 * Fluffer, i.e. expand generic code under assumptions about type
 * representations, which is essentially what the JIT does at runtime
 * in the generic CLR.  
 * -------------------------------------------------------------------- *)

let omap f opt = match opt with None -> None | Some x -> Some (f x)

let is_singular ilg tyenv = for_all (function ty when typ_is_Object ilg ty -> true | _ -> false) tyenv 
let replicate n x =   let rec go m = if m >= n then [] else x::go(m+1) in go 0

let ungenericize =  
  let t = Hashtbl.create 10 in 
  fun n ->
    let sym = '`' in
    if 
      String.contains n sym && 
      (* check what comes after the symbol is a number *)
      begin
        let m = String.rindex n sym in 
        let res = ref (m < String.length n - 1) in 
        for i = m + 1 to String.length n - 1 do
          res := !res && String.get n i >= '0' && String.get n i <= '9';
        done;
        !res
      end
    then 
      if Hashtbl.mem t n then Hashtbl.find t n 
      else 
        let pos = String.rindex n sym in 
        let res = String.sub n 0 pos in 
        (* HACK *)
        let res = res ^  String.sub n (pos + 1) (String.length n - (pos + 1)) in 
        Hashtbl.add t n res;
        res 
    else n
      

let mangle_name ilg n inst = 
  let n = ungenericize n in 
  if is_singular ilg inst then n
  else
    let mangle1 sofar act = 
      match act with 
      | ty when typ_is_Int32 ilg ty -> sofar^"_int32"
      | ty when typ_is_Int16 ilg ty -> sofar^"_int16"
      | ty when typ_is_SByte ilg ty -> sofar^"_int8"
      | ty when typ_is_Int64 ilg ty -> sofar^"_int64"
      | ty when typ_is_Object ilg ty -> sofar^"_O" 
      | ty -> 
	  dprint_endline ("*** nupp_erase.ml: Error when erasing non-uniform polymorphism: bad representation type argument to "^n);
	  dprint_string "*** Bad representation type is: ";
          output_typ stderr ty;
	  dprint_endline "";
          failwith "";
	  sofar^"_O"  in 
    fold_left mangle1 n inst

(* This gives the actual parameters for every generic variable in scope.  *)
type tyenv = genactuals

exception Bad_inst_read of tyenv * u16
let rec ufluff_typ (ilg:mscorlib_refs) tyenv typ = 
  try 
   match typ with 
  | Type_ptr t -> Type_ptr (ufluff_typ ilg tyenv t)
  | Type_fptr callsig -> Type_fptr (ufluff_callsig ilg tyenv callsig)
  | Type_byref t -> Type_byref (ufluff_typ ilg tyenv t)
  | Type_boxed cr -> Type_boxed (ufluff_tspec ilg tyenv cr)
  | Type_value ir -> Type_value (ufluff_tspec ilg tyenv ir)
  | Type_array(shape,ty) -> Type_array(shape,ufluff_typ ilg tyenv ty)
  | Type_tyvar v -> 
      begin 
        try inst_read tyenv v
        with Failure s -> ilg.typ_Object (* raise (Bad_inst_read (tyenv,v))  *)
      end
  | Type_other e -> failwith "nupp_erase.ml: no extensions should be present in the type algebra at this stage"
  | x -> x
    with e -> 
      dprintf1 "*** nupp_erase.ml: Internal error: Error while erasing non uniform pp from type, e = %s, " (Printexc.to_string e);  
      dprintf0 " type = ";  
      Ilprint.output_typ stderr typ;
      dprintf0 "\n";  
      (*F# rethrow(); F#*) raise e  
and ufluff_tspec ilg tyenv x =
  { tspecInst= mk_empty_gactuals;
    tspecTypeRef= ufluff_tref ilg (ufluff_inst ilg tyenv (inst_of_tspec x)) (tref_of_tspec x); }
and ufluff_tref ilg inst' x = 
  if inst' = [] then x else
  { x with trefNested=map ungenericize x.trefNested; trefName=mangle_name ilg x.trefName inst' }
and ufluff_inst ilg tyenv inst = map (ufluff_typ ilg tyenv) inst
and ufluff_callsig ilg tyenv x = 
  {x with callsigArgs=map (ufluff_typ ilg tyenv) x.callsigArgs;
          callsigReturn=ufluff_typ ilg tyenv x.callsigReturn }



(* -------------------------------------------------------------------- 
 * Modify method signatures, identities etc. based on type
 * token passing and type erasure.
 * -------------------------------------------------------------------- *)

(* Note that [cinst'] must be given to know the version *)
(* of the class being fluffed for the class where the method resides. *)
(* Also [minst'] must be given to know the version of the method being fluffed. *)

let ufluff_mref ilg tyenv cinst' minst' x = 
  { x with mrefParent=ufluff_tref ilg cinst' x.mrefParent;
	   mrefArity=0;
	   mrefName=mangle_name ilg x.mrefName minst';
	   mrefArgs=map (ufluff_typ ilg tyenv) x.mrefArgs;
	   mrefReturn=ufluff_typ ilg tyenv x.mrefReturn }

let ufluff_mspec ilg tyenv x = 
  try 
      let mref,ctyp,minst = dest_mspec x in 
      let ctyp' = ufluff_typ ilg tyenv ctyp in
      let cinst' = ufluff_inst ilg tyenv (inst_of_typ ctyp) in
      let minst' = ufluff_inst ilg tyenv minst in
      mk_mref_mspec_in_typ (ufluff_mref ilg (cinst'@minst') cinst' minst' mref,ctyp',mk_empty_gactuals)
    with e -> 
      dprintf1 "*** nupp_erase.ml: Internal error: Error while erasing non uniform pp from mspec, e = %s, " (Printexc.to_string e);  
      dprintf0 " mspec = ";  
      Ilprint.output_mspec stderr x;
      dprintf0 "\n";  
      (*F# rethrow(); F#*) raise e  

(* [minst'] must be given to know the version of the method being fluffed. *)
let ufluff_ospec ilg tyenv minst' (OverridesSpec(mref,ctyp)) = 
  let ctyp' = ufluff_typ ilg tyenv ctyp in
  let cinst' = ufluff_inst ilg tyenv (inst_of_typ ctyp) in
  OverridesSpec(ufluff_mref ilg (cinst'@minst') cinst' minst' mref,ctyp')

let ufluff_fspec ilg tyenv x = 
   let ctyp = enclosing_typ_of_fspec x in 
   let ctyp' = ufluff_typ ilg tyenv ctyp in
   let cinst' = ufluff_inst ilg tyenv (inst_of_typ ctyp) in
   { fspecFieldRef = { frefParent=ufluff_tref ilg cinst' x.fspecFieldRef.frefParent;
		       frefName=x.fspecFieldRef.frefName;
		       frefType=ufluff_typ ilg cinst' x.fspecFieldRef.frefType}; 
     fspecEnclosingType= ctyp' }

let ufluff_instr ilg tyenv instr  = 
  match instr with 
  | I_call (a,mr,b) -> [ I_call (a,ufluff_mspec ilg tyenv mr,b) ]
  | I_ldftn (mr) -> [ I_ldftn (ufluff_mspec ilg tyenv mr) ]
  | I_ldvirtftn (mr) -> [ I_ldvirtftn (ufluff_mspec ilg tyenv mr) ]
  | I_calli (a,mref,b) ->  [I_calli (a,ufluff_callsig ilg tyenv mref,b)] 
  | I_callvirt (a,mr,b) ->  [ I_callvirt (a,ufluff_mspec ilg tyenv mr,b) ]
  | I_callconstraint (a,ty,mr,b) ->  [ I_callconstraint (a,ufluff_typ ilg tyenv ty, ufluff_mspec ilg tyenv mr,b) ]
  | I_newobj (mr,b) -> [ I_newobj (ufluff_mspec ilg tyenv mr,b) ]
  | I_ldfld (a,b,fr) -> [I_ldfld (a,b,ufluff_fspec ilg tyenv fr)]
  | I_ldsfld (a,fr) ->   [I_ldsfld (a,ufluff_fspec ilg tyenv fr)]
  | I_ldflda fr ->    [I_ldflda (ufluff_fspec ilg tyenv fr)]

  | I_stfld (a,b,fr) ->  [I_stfld (a,b,ufluff_fspec ilg tyenv fr)]
  | I_stsfld (a,fr) ->  [I_stsfld (a,ufluff_fspec ilg tyenv fr)]

  (* castclass: this is "castclass w.r.t. repr. type" - *)
  (* getting correct cast semantics is taken care of in pp_erase. *)
  | I_castclass typ ->
      begin 
	let typ' = ufluff_typ ilg tyenv typ in 
	match typ' with 
	| Type_array _ 
	| Type_boxed _ ->   [ I_castclass typ']
	| Type_value _ ->   []  (* deliberate: generic interpretation for castclass over value type instantiations is "nop" *)
	| _ -> failwith "nupp_erase: cannot use castclass with this type"
      end

  (* isinst: this is "isinst w.r.t. to repr. type" - *)
  (* getting correct type semantics for isinst is taken care of in pp_erase. *)
  | I_isinst typ -> 
      begin 
	let typ' = ufluff_typ ilg tyenv typ in 
	match typ' with 
	| Type_array _
	| Type_boxed _ 
	| Type_value _ ->   [I_isinst typ']
	| _ -> failwith "nupp_erase: cannot isinst with this type"
      end
  | I_box typ -> 
      begin 
	let typ' = ufluff_typ ilg tyenv typ in 
	match typ' with 
	| Type_array _
	| Type_boxed _ ->   []  (* deliberate: generic interpretation for box over ref. type is "nop" *)
	| Type_value _ ->   [I_box typ']
	| _ -> failwith "nupp_erase: cannot box with this type"
      end
  | I_unbox typ ->  [I_unbox (ufluff_typ ilg tyenv typ)]
  | I_unbox_any typ ->
      begin 
	let typ' = ufluff_typ ilg tyenv typ in 
	match typ' with 
	| Type_array _ 
	| Type_boxed _ ->  if typ_is_Object ilg typ' then  [] else  [I_castclass typ']
	| _ ->    [I_unbox typ'; I_ldobj (Aligned,Nonvolatile,typ')]
      end

  (* ldtoken: this is "ldtoken for the repr. type" - *)
  (* loading the correct RTT is *)
  (* taken care of in pp_erase. *)
  | I_ldtoken tokspec -> 
      let ts' = 
	match tokspec with 
	| Token_type typ -> Token_type (ufluff_typ ilg tyenv typ)
	| Token_method mr -> Token_method (ufluff_mspec ilg tyenv mr)
	| Token_field fr -> Token_field (ufluff_fspec ilg tyenv fr) in 
       [I_ldtoken ts' ]

  | I_ldelem_any (shape,typ) ->  
      begin 
	let typ' = ufluff_typ ilg tyenv typ in 
	match typ' with 
	| ty when shape = sdshape && typ_is_SByte ilg ty ->  [I_ldelem DT_I1]
	| ty when shape = sdshape && typ_is_Int16 ilg ty ->  [I_ldelem DT_I2]
	| ty when shape = sdshape && typ_is_Int32 ilg ty ->  [I_ldelem DT_I4]
	| ty when shape = sdshape && typ_is_Int64 ilg ty ->  [I_ldelem DT_I8]
	| ty when shape = sdshape && typ_is_Byte ilg ty ->  [I_ldelem DT_U1]
	| ty when shape = sdshape && typ_is_UInt16 ilg ty ->  [I_ldelem DT_U2]
	| ty when shape = sdshape && typ_is_UInt32 ilg ty ->  [I_ldelem DT_U4]
	| ty when shape = sdshape && typ_is_UInt64 ilg ty ->  [I_ldelem DT_U8]
	| ty when shape = sdshape && typ_is_UIntPtr ilg ty ->  [I_ldelem DT_I]
	| ty when shape = sdshape && typ_is_IntPtr ilg ty ->  [I_ldelem DT_I]
	| Type_array _
	| Type_boxed _ when shape = sdshape ->  [I_ldelem DT_REF]
	| _ -> 
            (* For value types we get rid of the non-generic opcode *)
            let ty' = ufluff_typ ilg tyenv typ in 
            [ I_ldelema (NormalAddress,shape,ty'); mk_normal_ldobj(ty') ]
      end

  | I_stelem_any (shape,typ) ->  
      begin 
	let typ' = ufluff_typ ilg tyenv typ in 
	match typ' with 
	| ty when shape = sdshape && typ_is_SByte ilg ty ->  [I_stelem DT_I1]
	| ty when shape = sdshape && typ_is_Int16 ilg ty ->  [I_stelem DT_I2]
	| ty when shape = sdshape && typ_is_Int32 ilg ty ->  [I_stelem DT_I4]
	| ty when shape = sdshape && typ_is_Int64 ilg ty ->  [I_stelem DT_I8]
	| ty when shape = sdshape && typ_is_Byte ilg ty ->  [I_stelem DT_I1]
	| ty when shape = sdshape && typ_is_UInt16 ilg ty ->  [I_stelem DT_I2]
	| ty when shape = sdshape && typ_is_UInt32 ilg ty ->  [I_stelem DT_I4]
	| ty when shape = sdshape && typ_is_UInt64 ilg ty ->  [I_stelem DT_I8]
	| ty when shape = sdshape && typ_is_UIntPtr ilg ty ->  [I_stelem DT_I]
	| ty when shape = sdshape && typ_is_IntPtr ilg ty ->  [I_stelem DT_I]
	| Type_array _
	| Type_boxed _ when shape = sdshape  ->  [I_stelem DT_REF]
	| _ when shape = sdshape -> failwith "Error: .NET v2.0 instruction I_stelem on a value type can not yet be erased"
	| _ ->  [I_stelem_any (shape,ufluff_typ ilg tyenv typ)]
      end

  | I_newarr (shape,typ) ->   [I_newarr (shape,ufluff_typ ilg tyenv typ)]  

  | I_ldelema (ro,shape,typ) ->   [I_ldelema (ro,shape,ufluff_typ ilg tyenv typ)] 


  | I_stobj (a,b,typ) ->  
      begin 
	let typ' = ufluff_typ ilg tyenv typ in 
	match typ' with 
	| ty when typ_is_SByte ilg ty ->  [I_stind (a,b, DT_I1)]
	| ty when typ_is_Int16 ilg ty ->  [I_stind (a,b, DT_I2)]
	| ty when typ_is_Int32 ilg ty ->  [I_stind (a,b, DT_I4)]
	| ty when typ_is_Int64 ilg ty ->  [I_stind (a,b, DT_I8)]
	| ty when typ_is_Byte  ilg ty ->  [I_stind (a,b, DT_U1)]
	| ty when typ_is_UInt16 ilg ty ->  [I_stind (a,b, DT_U2)]
	| ty when typ_is_UInt32 ilg ty ->  [I_stind (a,b, DT_U4)]
	| ty when typ_is_UInt64 ilg ty ->  [I_stind (a,b, DT_U8)]
	| Type_array _
	| Type_boxed _ ->   [I_stind (a,b, DT_REF)]
	| _ ->  [I_stobj (a,b,typ')]
      end

  | I_ldobj (a,b,typ) ->  
      begin 
	let typ' = ufluff_typ ilg tyenv typ in 
	match typ' with 
	| ty when typ_is_SByte ilg ty ->  [I_ldind (a,b,DT_I1)]
	| ty when typ_is_Int16 ilg ty ->  [I_ldind (a,b, DT_I2)]
	| ty when typ_is_Int32 ilg ty ->  [I_ldind (a,b, DT_I4)]
	| ty when typ_is_Int64 ilg ty ->  [I_ldind (a,b, DT_I8)]
	| ty when typ_is_Byte ilg ty ->  [I_ldind (a,b, DT_U1)]
	| ty when typ_is_UInt16 ilg ty ->  [I_ldind (a,b, DT_U2)]
	| ty when typ_is_UInt32 ilg ty ->  [I_ldind (a,b, DT_U4)]
	| ty when typ_is_UInt64 ilg ty ->  [I_ldind (a,b, DT_U8)]
	| Type_array _
	| Type_boxed _ ->   [mk_normal_ldind DT_REF]
	| _ ->  [I_ldobj (a,b,typ')]
      end

  | I_cpobj typ ->  
      begin 
	let typ' = ufluff_typ ilg tyenv typ in 
	match typ' with 
	| Type_array _
	| Type_boxed _ ->   (mk_normal_cpind DT_REF)
	| _ ->  [I_cpobj typ']
      end

  | I_initobj typ ->  
      begin 
	let typ' = ufluff_typ ilg tyenv typ in 
	match typ' with 
	| Type_array _
	| Type_boxed _ ->   [ I_arith AI_ldnull; mk_normal_stind DT_REF ]
	| Type_ptr _ ->  [ I_arith (AI_ldc (DT_I4,NUM_I4 (i32_zero))); 
			   I_arith (AI_conv DT_I); 
			   mk_normal_stind DT_I ]
	| _ ->  [I_initobj typ']
      end

  | _ ->  [instr]

let ufluff_ilmbody ilg tyenv il = 
  let newcode = 
    topcode_instr2instrs 
      (fun i -> ufluff_instr ilg tyenv i) 
      il.ilCode in 
  {il with ilCode=newcode;
           ilLocals = locals_typ2typ (ufluff_typ ilg tyenv) il.ilLocals}

let ufluff_mbody ilg tyenv mbody = 
    mbody_ilmbody2ilmbody (ufluff_ilmbody ilg tyenv) mbody

(* -------------------------------------------------------------------- 
 * Modify a method definition
 *
 * There is a problem with storing type tokens in fields - they are
 * not needed in unboxed value classes, but, strictly speaking
 * are needed in boxed ones...
 * -------------------------------------------------------------------- *)

let add_tyenvs t1 t2 = inst_add t1 t2

let ufluff_mdef_per_inst ilg class_tyenv md method_tyenv  = 
 try 
  let tyenv = add_tyenvs class_tyenv method_tyenv in
  let newkind = 
    try 
      match md.mdKind with 
      | MethodKind_virtual vinfo -> 
	  MethodKind_virtual 
	    {vinfo with virtOverrides=
              match vinfo.virtOverrides with 
		None -> None 
              | Some p -> Some (ufluff_ospec ilg tyenv method_tyenv p)}
      | k -> k 
    with e -> 
      dprint_string "*** nupp_erase.ml: Internal error: Error while erasing non uniform pp from method-parent of method ";  dprint_endline md.mdName;
      (*F# rethrow(); F#*) raise e  in 
  let res = 
   let b' = ufluff_mbody ilg tyenv md.mdBody in 
   {md with mdName= (if is_singular ilg method_tyenv then md.mdName else mangle_name ilg md.mdName method_tyenv);
            mdGenericParams=[];
            mdBody=b';
            mdParams = params_typ2typ (ufluff_typ ilg tyenv) md.mdParams ;
            mdReturn = return_typ2typ (ufluff_typ ilg tyenv) md.mdReturn ;
            mdKind=newkind } in 
  [ res ]
 with e -> 
   dprint_string "*** nupp_erase.ml: Error while erasing non uniform pp from mbody of method "; dprint_endline md.mdName;
   (*F# rethrow(); F#*) raise e

(* -------------------------------------------------------------------- 
 * Generate all poss. combinations of representations for a given set of type variables 
 * The basis for one "generate ilg" is an instantiation giving a mapping for types. 
 * -------------------------------------------------------------------- *)

let generate ilg f genparams = f (map (fun _ -> ilg.typ_Object) genparams)

let ufluff_mdef ilg class_tyenv md  =
  generate ilg (ufluff_mdef_per_inst ilg class_tyenv md) md.mdGenericParams

let ufluff_interior_mref ilg td class_tyenv x = 
  try 
  {x with mrefParent= tref_of_typ (ufluff_typ ilg class_tyenv (Type_boxed (mk_tspec(x.mrefParent, class_tyenv))));
          mrefArgs= map (ufluff_typ ilg class_tyenv) x.mrefArgs; 
          mrefReturn=ufluff_typ ilg class_tyenv x.mrefReturn }
 with e -> 
   dprintf1 "*** nupp_erase.ml: ufluff_interior_mref, name = %s\n" (name_of_mref x);
   (*F# rethrow(); F#*) raise e


let ufluff_pdef ilg td class_tyenv p =
  try 
  { p with
    propSet = omap (ufluff_interior_mref ilg td class_tyenv) p.propSet;
    propGet = omap (ufluff_interior_mref ilg td class_tyenv) p.propGet;
    propType = ufluff_typ ilg class_tyenv p.propType;
    propArgs = map (ufluff_typ ilg class_tyenv) p.propArgs;
    propCustomAttrs = cattrs_typ2typ (ufluff_typ ilg class_tyenv) p.propCustomAttrs }
 with e -> 
   dprint_string "*** nupp_erase.ml: Error while erasing non uniform pp from property "; dprint_endline p.propName;
   (*F# rethrow(); F#*) raise e

let ufluff_pdefs ilg td class_tyenv pdefs = 
  mk_properties (map (ufluff_pdef ilg td class_tyenv) (dest_pdefs pdefs))

let ufluff_edef ilg td class_tyenv e =
  { e with
    eventType = omap (ufluff_typ ilg class_tyenv) e.eventType;
    eventAddOn = ufluff_interior_mref ilg td class_tyenv e.eventAddOn;
    eventRemoveOn = ufluff_interior_mref ilg td class_tyenv e.eventRemoveOn;
    eventFire = omap (ufluff_interior_mref ilg td class_tyenv) e.eventFire;
    eventOther = map (ufluff_interior_mref ilg td class_tyenv) e.eventOther;
    eventCustomAttrs = cattrs_typ2typ (ufluff_typ ilg class_tyenv) e.eventCustomAttrs }

let ufluff_mimpl_per_inst ilg class_tyenv mimpl method_tyenv =
  let (OverridesSpec(mref,ctyp) as ovspec) = ufluff_ospec ilg class_tyenv method_tyenv mimpl.mimplOverrides in 
  let tyenv = add_tyenvs class_tyenv method_tyenv in
  let ovby = ufluff_mspec ilg tyenv mimpl.mimplOverrideBy in 
  (* remap the method impl's signature - ok because there are no generics left *)
  let ovby = 
    let _,typ,minst = dest_mspec ovby in 
    mk_mref_mspec_in_typ({mref with mrefName= name_of_mspec ovby; mrefParent=tref_of_typ typ },typ,minst) in
  [ { mimplOverrides = ovspec;
      mimplOverrideBy = ovby } ]

let ufluff_mimpl ilg class_tyenv mimpl =
  try 
    let somewhatFakedGenericParamsOfOverridingMethod = 
      (replicate (genarity_of_mspec mimpl.mimplOverrideBy) (mk_simple_gparam "T")) in 
    generate ilg (ufluff_mimpl_per_inst ilg class_tyenv mimpl) somewhatFakedGenericParamsOfOverridingMethod
 with e -> 
   dprint_string "*** nupp_erase.ml:  internal error:, mimpl.Name = "; dprint_endline (name_of_mspec mimpl.mimplOverrideBy);
   (*F# rethrow(); F#*) raise e

let ufluff_mimpls ilg  class_tyenv mimpls = 
  mk_mimpls (concat (map (ufluff_mimpl ilg class_tyenv) (dest_mimpls mimpls)))

let ufluff_edefs ilg td class_tyenv edefs = 
  mk_events (map (ufluff_edef ilg td class_tyenv) (dest_edefs edefs))

let rec ufluff_tdef_per_inst ilg td class_tyenv = 
  let statics = is_singular ilg class_tyenv in 
  try 
    (* First produce the uniform version of the class *)
    [ {td with tdName=(if statics then ungenericize td.tdName else mangle_name ilg td.tdName class_tyenv);
              tdGenericParams=[]; 
              tdExtends=omap (ufluff_typ ilg class_tyenv) td.tdExtends; 
              tdImplements=map (ufluff_typ ilg class_tyenv) td.tdImplements; 
              tdNested=(if statics then tdefs_tdef2tdefs (ufluff_tdef ilg) td.tdNested else mk_tdefs []);
              tdMethodDefs= mdefs_mdef2mdefs (ufluff_mdef ilg class_tyenv)  (filter_mdefs (fun md -> statics || md.mdKind <> MethodKind_static) td.tdMethodDefs); 
              tdMethodImpls= ufluff_mimpls ilg class_tyenv td.tdMethodImpls;
              tdFieldDefs= fdefs_typ2typ (ufluff_typ ilg class_tyenv) (filter_fdefs (fun fd ->statics || not fd.fdStatic) td.tdFieldDefs);
              tdProperties= ufluff_pdefs ilg td.tdName class_tyenv (filter_pdefs (fun pd ->statics || pd.propCallconv <> CC_static) td.tdProperties);
              tdEvents= ufluff_edefs ilg td class_tyenv td.tdEvents;
             }  ]
  with e -> 
    dprint_endline ("*** nupp_erase.ml: internal error: ufluff_tdef_per_inst ilg, td.tdName = "^  td.tdName); 
   Ilprint.output_tdef stderr td;
    (*F# rethrow(); F#*) raise e

and ufluff_tdef ilg td  = generate ilg (ufluff_tdef_per_inst ilg td) td.tdGenericParams

let conv_module ilg modul =  module_tdefs2tdefs (tdefs_tdef2tdefs (ufluff_tdef ilg)) modul

