/** \file
 *  
 *  Copyright (c) 2014 by Travis Gockel. All rights reserved.
 *
 *  This program is free software: you can redistribute it and/or modify it under the terms of the Apache License
 *  as published by the Apache Software Foundation, either version 2 of the License, or (at your option) any later
 *  version.
 *
 *  \author Travis Gockel (travis@gockelhut.com)
**/
#include "test.hpp"

#include <jsonv/char_convert.hpp>

#include <sstream>

using jsonv::detail::decode_error;

template <std::size_t N>
static std::string string_decode_static(const char (& data)[N],
                                        jsonv::parse_options::encoding encoding = jsonv::parse_options::encoding::utf8
                                       )
{
    auto decoder = jsonv::detail::get_string_decoder(encoding);
    // Use N-1 for length to not include the '\0' at the end
    return decoder(jsonv::string_view(data, N-1));
}

TEST(string_decode_invalid_escape_sequence)
{
    ensure_throws(jsonv::detail::decode_error, string_decode_static("\\y"));
}

TEST(string_decode_unchanged)
{
    ensure_eq("Hello!", string_decode_static("Hello!"));
}

TEST(string_decode_sigils)
{
    ensure_eq("\t\b\f\n\r", string_decode_static("\\t\\b\\f\\n\\r"));
}

TEST(string_decode_utf_one_char)
{
    ensure_eq("\xe2\x98\xa2", string_decode_static("\\u2622"));
}

TEST(string_decode_utf_char_starts)
{
    ensure_eq("\xe2\x98\xa2normal text", string_decode_static("\\u2622normal text"));
}

TEST(string_decode_utf_char_ends)
{
    ensure_eq("normal text\xe2\x98\xa2", string_decode_static("normal text\\u2622"));
}

TEST(string_decode_utf_char_bookends)
{
    ensure_eq("\xe2\x98\xa2normal text\xe2\x9d\xa4", string_decode_static("\\u2622normal text\\u2764"));
}

TEST(string_decode_utf_char_surrounded)
{
    ensure_eq("normal\xe2\x98\xa2text", string_decode_static("normal\\u2622text"));
}

TEST(string_decode_utf_many_chars)
{
    ensure_eq("\xe2\x9d\xa4 \xe2\x98\x80 \xe2\x98\x86 \xe2\x98\x82 \xe2\x98\xbb \xe2\x99\x9e \xe2\x98\xaf \xe2\x98\xad \xe2\x98\xa2 \xe2\x82\xac \xe2\x86\x92 \xe2\x98\x8e \xe2\x9d\x84 \xe2\x99\xab \xe2\x9c\x82 \xe2\x96\xb7 \xe2\x9c\x87 \xe2\x99\x8e \xe2\x87\xa7 \xe2\x98\xae \xe2\x99\xbb \xe2\x8c\x98 \xe2\x8c\x9b \xe2\x98\x98",
              string_decode_static("\\u2764 \\u2600 \\u2606 \\u2602 \\u263b \\u265e \\u262f \\u262d \\u2622 \\u20ac \\u2192 \\u260e \\u2744 \\u266b \\u2702 \\u25b7 \\u2707 \\u264e \\u21e7 \\u262e \\u267b \\u2318 \\u231b \\u2618"));
}

TEST(string_decode_annoying_mix)
{
    string_decode_static(R"(\/\\\"\uCAFE\uBABE\uAB98\uFCDE\ubcda\uef4A\b\f\n\r\t`1~!@#$%^&*()_+-=[]{}|;:',./<>?)");
}

// A bunch of surrogate pairs generated by Python
#define JSONV_TEST_SURROGATE_PAIRS(item)        \
    item("\xf0\x90\x80\x80", R"(\ud800\udc00)") \
    item("\xf0\x97\x9f\xa0", R"(\ud81d\udfe0)") \
    item("\xf2\xba\xbe\x85", R"(\udaab\udf85)") \
    item("\xf0\xba\xad\xbe", R"(\ud8aa\udf7e)") \
    item("\xf1\xbb\xb7\xb0", R"(\ud9af\uddf0)") \
    item("\xf0\xb7\xbc\xb4", R"(\ud89f\udf34)") \
    item("\xf0\x9c\xa9\xa3", R"(\ud832\ude63)") \
    item("\xf3\x81\xbd\xad", R"(\udac7\udf6d)") \
    item("\xf3\x97\xa6\xa0", R"(\udb1e\udda0)") \
    item("\xf4\x8a\x91\xbe", R"(\udbe9\udc7e)") \
    item("\xf2\xb5\xb4\xba", R"(\uda97\udd3a)") \
    item("\xf0\xb7\x9e\xb4", R"(\ud89d\udfb4)") \
    item("\xf3\xa8\x8e\x8c", R"(\udb60\udf8c)") \
    item("\xf2\x8b\x98\x85", R"(\ud9ed\ude05)") \
    item("\xf3\x99\xae\x9c", R"(\udb26\udf9c)") \
    item("\xf1\xaa\xa9\x96", R"(\ud96a\ude56)") \
    item("\xf1\xb4\x98\x8a", R"(\ud991\ude0a)") \
    item("\xf0\x94\xab\x97", R"(\ud812\uded7)") \
    item("\xf2\x80\xbf\xb8", R"(\ud9c3\udff8)") \
    item("\xf1\xb8\x8d\x8d", R"(\ud9a0\udf4d)") \
    item("\xf3\x95\x84\xaa", R"(\udb14\udd2a)") \
    item("\xf0\xab\xbf\x8b", R"(\ud86f\udfcb)") \
    item("\xf1\x99\x86\x82", R"(\ud924\udd82)") \
    item("\xf2\x8f\x87\xa0", R"(\ud9fc\udde0)") \
    item("\xf2\x85\x8b\x97", R"(\ud9d4\uded7)") \
    item("\xf2\xb6\xa5\xaa", R"(\uda9a\udd6a)") \
    item("\xf2\x9f\xb4\xa7", R"(\uda3f\udd27)") \
    item("\xf3\x91\x9b\x9f", R"(\udb05\udedf)") \
    item("\xf1\xab\xbb\x9c", R"(\ud96f\udedc)") \
    item("\xf3\xb5\xb3\xa9", R"(\udb97\udce9)") \
    item("\xf4\x82\x95\x93", R"(\udbc9\udd53)") \
    item("\xf1\xb1\xa9\xb5", R"(\ud986\ude75)") \
    item("\xf2\xad\x85\xbf", R"(\uda74\udd7f)") \
    item("\xf1\x8e\x8f\xa2", R"(\ud8f8\udfe2)") \
    item("\xf1\x9e\xba\x9a", R"(\ud93b\ude9a)") \
    item("\xf2\x90\xac\xb3", R"(\uda02\udf33)") \
    item("\xf3\x9d\xa7\x82", R"(\udb36\uddc2)") \
    item("\xf2\xa1\xab\xa6", R"(\uda46\udee6)") \
    item("\xf2\x94\x98\x97", R"(\uda11\ude17)") \
    item("\xf3\xa0\x87\xa2", R"(\udb40\udde2)") \
    item("\xf2\x9b\xa7\xb3", R"(\uda2e\uddf3)") \
    item("\xf2\xa7\xab\xaf", R"(\uda5e\udeef)") \
    item("\xf1\xb3\x90\xa1", R"(\ud98d\udc21)") \
    item("\xf2\xbb\x88\x81", R"(\udaac\ude01)") \
    item("\xf3\xab\x9e\xb6", R"(\udb6d\udfb6)") \
    item("\xf4\x8e\xb3\xa1", R"(\udbfb\udce1)") \
    item("\xf4\x8a\x95\x99", R"(\udbe9\udd59)") \
    item("\xf0\x99\xbe\xb9", R"(\ud827\udfb9)") \
    item("\xf0\xa6\x9c\x85", R"(\ud859\udf05)") \
    item("\xf1\x97\xb3\xbf", R"(\ud91f\udcff)") \
    item("\xf1\x9a\xbc\xbb", R"(\ud92b\udf3b)") \

TEST(string_decode_unicode_surrogates_valid)
{
    #define JSONV_TEST_GEN_DECODE_EQ(utf8encoding_, jsonencoding_)     \
        ensure_eq(utf8encoding_, string_decode_static(jsonencoding_)); \
    
    JSONV_TEST_SURROGATE_PAIRS(JSONV_TEST_GEN_DECODE_EQ)
}

TEST(string_decode_unicode_surrogates_invalid_end_immediate)
{
    ensure_throws(decode_error, string_decode_static(R"(\ud92b)"));
}

TEST(string_decode_unicode_surrogates_invalid_non_numeric_following)
{
    ensure_throws(decode_error, string_decode_static(R"(\ud92b____i____)"));
    ensure_throws(decode_error, string_decode_static(R"(\ud92b,.//<<>>/)"));
    ensure_throws(decode_error, string_decode_static(R"(\ud92b??!!897123)"));
    ensure_throws(decode_error, string_decode_static(R"(\ud92b__)"));
}

TEST(string_decode_unicode_surrogates_invalid_incomplete_escapes)
{
    ensure_throws(decode_error, string_decode_static(R"(\ud92b\u)"));
    ensure_throws(decode_error, string_decode_static(R"(\ud92b\ud)"));
    ensure_throws(decode_error, string_decode_static(R"(\ud92b\udc)"));
    ensure_throws(decode_error, string_decode_static(R"(\ud92b\udcb)"));
}

TEST(string_decode_unicode_surrogates_invalid_non_hex)
{
    ensure_throws(decode_error, string_decode_static(R"(\ud92b\uPIKA)"));
    ensure_throws(decode_error, string_decode_static(R"(\ud92b\ucattle)"));
    ensure_throws(decode_error, string_decode_static(R"(\ud92b\u_________)"));
    ensure_throws(decode_error, string_decode_static(R"(\ud92b\ujust bogus)"));
}

TEST(string_decode_unicode_surrogates_invalid_not_low_follower)
{
    ensure_throws(decode_error, string_decode_static(R"(\ud92b\ucccc)"));
    ensure_throws(decode_error, string_decode_static(R"(\ud92b\ubeefy)"));
    ensure_throws(decode_error, string_decode_static(R"(\ud92b\ud800)"));
    ensure_throws(decode_error, string_decode_static(R"(\ud92b\udbff\udc00)"));
}

static std::string string_encode_static(const std::string& source)
{
    std::ostringstream ss;
    jsonv::detail::string_encode(ss, source);
    return ss.str();
}

TEST(string_encode_surrogates_valid)
{
    #define JSONV_TEST_GEN_ENCODE_EQ(utf8encoding_, jsonencoding_)     \
        ensure_eq(jsonencoding_, string_encode_static(utf8encoding_)); \
    
    JSONV_TEST_SURROGATE_PAIRS(JSONV_TEST_GEN_ENCODE_EQ)
}

#define JSONV_TEST_CESU8_ENCODINGS(item) \
    item("\\u0ff5", "\xe0\xbf\xb5")      \
    item("\\u091b", "\xe0\xa4\x9b")      \
    item("\\uc486", "\xec\x92\x86")      \
    item("\\u88cf", "\xe8\xa3\x8f")      \
    item("\\u282e", "\xe2\xa0\xae")      \
    item("\\u3bff", "\xe3\xaf\xbf")      \
    item("\\ub469", "\xeb\x91\xa9")      \
    item("\\ucb0b", "\xec\xac\x8b")      \
    item("\\ufab2", "\xef\xaa\xb2")      \
    item("\\u0044", "D")                 \
    item("\\ufdba", "\xef\xb6\xba")      \
    item("\\u3aa5", "\xe3\xaa\xa5")      \
    item("\\uaffe", "\xea\xbf\xbe")      \
    item("\\u3500", "\xe3\x94\x80")      \
    item("\\ubaa5", "\xeb\xaa\xa5")      \
    item("\\ud1b5", "\xed\x86\xb5")      \
    item("\\u4ff9", "\xe4\xbf\xb9")      \
    item("\\u7b08", "\xe7\xac\x88")      \
    item("\\u2044", "\xe2\x81\x84")      \
    item("\\uab55", "\xea\xad\x95")      \
    item("\\u7944", "\xe7\xa5\x84")      \
    item("\\uf513", "\xef\x94\x93")      \
    item("\\ue2dc", "\xee\x8b\x9c")      \
    item("\\ue072", "\xee\x81\xb2")      \
    item("\\ub100", "\xeb\x84\x80")      \
    item("\\u6868", "\xe6\xa1\xa8")      \
    item("\\u907f", "\xe9\x81\xbf")      \
    item("\\ue5ef", "\xee\x97\xaf")      \
    item("\\ucb23", "\xec\xac\xa3")      \
    item("\\u2283", "\xe2\x8a\x83")      \
    item("\\uf365", "\xef\x8d\xa5")      \
    item("\\u53d0", "\xe5\x8f\x90")      \
    item("\\u5605", "\xe5\x98\x85")      \
    item("\\u1ad2", "\xe1\xab\x92")      \
    item("\\u0279", "\xc9\xb9")          \
    item("\\u498b", "\xe4\xa6\x8b")      \
    item("\\u8463", "\xe8\x91\xa3")      \
    item("\\u48bb", "\xe4\xa2\xbb")      \
    item("\\u1bd1", "\xe1\xaf\x91")      \
    item("\\uf963", "\xef\xa5\xa3")      \
    item("\\u8383", "\xe8\x8e\x83")      \
    item("\\u4d63", "\xe4\xb5\xa3")      \
    item("\\udced", "\xed\xb3\xad")      \
    item("\\u5410", "\xe5\x90\x90")      \
    item("\\ucac8", "\xec\xab\x88")      \
    item("\\ucacb", "\xec\xab\x8b")      \
    item("\\ubc82", "\xeb\xb2\x82")      \
    item("\\u919e", "\xe9\x86\x9e")      \
    item("\\ua2ad", "\xea\x8a\xad")      \
    item("\\ud89c", "\xed\xa2\x9c")      \

TEST(string_decode_cesu8)
{
    #define JSONV_TEST_GEN_ENCODE_CESU8_EQ(jsonencoding_, cesu8encoding_)                                      \
        ensure_eq(string_decode_static(jsonencoding_, jsonv::parse_options::encoding::cesu8), cesu8encoding_); \
    
    JSONV_TEST_CESU8_ENCODINGS(JSONV_TEST_GEN_ENCODE_CESU8_EQ)
}

TEST(string_decode_long_utf8_sequences)
{
    // A couple of very long, but valid UTF-8 sequences
    string_decode_static("\xf8\x80\x80\x80\x80");
    string_decode_static("\xfc\x80\x80\x80\x80\x80");
}

TEST(string_decode_short_utf8_sequence)
{
    ensure_throws(decode_error, string_decode_static("\x80 is not 1 byte long"));
    ensure_throws(decode_error, string_decode_static("\xc0 is not 2 bytes long"));
    ensure_throws(decode_error, string_decode_static("\xe0\x80 is not 3 bytes long"));
    ensure_throws(decode_error, string_decode_static("\xf0\x80 is not 4 bytes long"));
}

TEST(string_decode_invalid_utf8_start)
{
    ensure_throws(decode_error, string_decode_static("\xfe is not a UTF-8 start"));
}
