/************************************************************************
*																		*
* 					Jim Schimpf - Pandora Products 						*
*					Copyright 2002 Jim Schimpf							*
*  																		*
*					MAC PROJECT BUILDER LEXICAL ANALYZER				*
*																		*
*		Module:CpbxLexFile.cp											*
*																		*
*	Version		Date	Person  	Description							*
*-----------------------------------------------------------------------*
*	2.4		28-Jul-2002	J.Schimpf	Convert to PBXPROJ parser			*
*	2.5		30-Jul-2002 J.Schimpf	Make { = ( and } = )				*
*	2.6		31-Jul-2002 J.Schimpf	Add code to return unknown puncts	*
*									as part of name tokens				*
*	2.7		31-Jul-2002 J.Schimpf	Add code to pass _,- and . as		*
*									part of names						*
*	2.8		31-Jul-2002 J.Schimpf	Fix partially bracked initializer	*
*									by bracketing compound initializer	*
*									elements							*
*																		*
*	DESCRIPTION:													    *
*		Converts file to a stream of tokens								*
*		TOKEN := [<WS chars>]<Contigious characters = TOKEN>[<WS>]		*
*				or <QUOTE><charcters with embedded spaces><QUOTE>		*
*				 Maximum # characters in token in MAX_TOK				*
*		Characters in a line following <COMMENT_CHAR> are ignored to EOL*
*		NOTE: Files are treated as lines and TOKENS cannot be extend	*
*		over line breaks												*
*																		*
*	Methods:															*
*																		*
*	Name					Description 								*
*-----------------------------------------------------------------------*
*																		*
************************************************************************/

/******* INCLUDE FILES ******/

#include "CpbxLexFile.h"
#include <ctype.h>

// *** PBXPROJ TOKEN LIST ***

LEX_TOKEN ctlist[] = 
{
		//{P_TOK_PATH,		"/"},
		{P_TOK_OBJ_START,	"{"},
		{P_TOK_OBJ_END,		"}"},
		{P_TOK_END,			";"},
		{P_TOK_LINK,		"="},
		{P_TOK_LIST,		","},
};

const int SIZE_CTLIST = sizeof( ctlist ) / sizeof( LEX_TOKEN );
const int NO_PENDING = -1; 

/***********************************************************************
*
*  init()	- Initializer for token run
*
*	INPUT:	cnt		- # Lexical tokens to be found
*			tlist	- List of tokens
*
*	OUTPUT:	NONE
*			Initialize all internal variables to default states
*
***********************************************************************/

void CpbxLexFile::init(void)
{

	// Init Lexical analyzer for this run

	CLexFile::init( SIZE_CTLIST,ctlist );

	// initialize current token stuff

	tok_psn = NO_PENDING;
	
	unget = false;
}	

/***********************************************************************
*		TOKEN FILE I/O OPERATIONS
************************************************************************/
/***********************************************************************
*
* bool get_token( char *val )	- Get the next token from the
*												  file
*
*	INPUT:	val	- Returned token, NOTE: Must have TOK_MAX+1 spaces
*
*	OUTPUT:	token type
*
***********************************************************************/

int CpbxLexFile::get_token( char *token )
{
	int type;
	
	// (0) Check for UNGET and do that and reset unget flag
	
	if( unget )
	{
		type = unget_type;
		strcpy(token,unget_val);
		unget = false;
		return( type );
	}

	// (1) Do we have something pending ?

	if( tok_psn != NO_PENDING )
	{
		type = this->get_pending_token( token );
		return( type );
	}

	// (2) Ok, we have nothing pending so get the next thing

	type = CLexFile::get_token( token );
	unget_type = type;
	strcpy(unget_val,token);
	
	switch( type )
	{
		case LEX_BUF_TOK:
							break;		// Parse further
							
		case LEX_QUOT_TOK:	return( P_TOK_NAME );	// Return as a name
							
		default:			return( type );
	}

	// (3) We have a name so set it into the pending slot and
	// re-extract what is left

	tok_psn = 0;
	strcpy(cur_token,token );

	type = this->get_pending_token( token );
	return( type );
}

/***********************************************************************
*
* void unget_token( )	- Unget the last token received 
*
*	INPUT:	NONE
*
*	OUTPUT:	NONE
*		Marked to allow re-get of the last token received
*
***********************************************************************/

void CpbxLexFile::unget_token( void )
{
	unget = true;
}

/***********************************************************************
*
* bool get_char(  )		- Get next character from pending token
*
*	INPUT:	NONE
*
*	OUTPUT:	NEXT character from pending token or TOK_EOL if no pending
*
***********************************************************************/

int CpbxLexFile::get_char(  )
{
	int ch;
	
	// If nothing pending then return EOL
	if( tok_psn == NO_PENDING )
		return( TOK_BUF_EOL );
	
	// Ok we have something so return it
	
	ch = cur_token[tok_psn++];
	if( iscntrl( ch ) )
	{
		tok_psn = NO_PENDING;
		return( TOK_BUF_EOL );
	}
	else
		return( ch );
}

/***********************************************************************
*
* bool unget_char(   )	- Unget last character from pending buffer
*
*	INPUT:	NONE
*
*	OUTPUT:	NONE
*			Back up pending buffer pointer
*
***********************************************************************/

void CpbxLexFile::unget_char(  )
{
	if( tok_psn != NO_PENDING )
	{
		tok_psn--;
		if( tok_psn < 0 )
			tok_psn = 0;
	}
}

/****************** INTERNAL PRIVATE OPERATIONS ************************/
/***********************************************************************
*
* bool get_pending_token( char *val )	- Get the next token from the
*												  pending token
*
*	INPUT:	val	- Returned token, NOTE: Must have TOK_MAX+1 spaces
*
*	OUTPUT:	Token type
*
***********************************************************************/

int CpbxLexFile::get_pending_token( char *token )
{
	int type;
	int i;
	
	// (1) Scan through and stop on first non-alpha thing

	for( i=0; i< MAX_C_TOKEN; i++ )
	{
		if( isalnum(cur_token[tok_psn]) || cur_token[tok_psn] == '_' 
                                        || cur_token[tok_psn] == '.'
                                        || cur_token[tok_psn] == '-'
										|| cur_token[tok_psn] == '$' )	// REF:JS09272003
		{
			token[i] = cur_token[tok_psn++];
 		}
		else
			break;			// Quit on NON-ALPHA
	}

	// (2) If i == 0 then we found a non-alpha on first pass and
	// we are dealing with that

	if( i != 0 )
	{
		token[i] = '\0';		// Terminate the string and
		type = P_TOK_NAME;		// return that it's a ID
 	}
	else
	{

		// (3) We have hit a punct mark so determine which is was and return
		// that

		
		token[0] = cur_token[tok_psn++];
		token[1] = '\0';

		switch( token[0] ) {
            
            case '(':
			case '{':		type = P_TOK_OBJ_START;
							break;

            case ')':
			case '}':		type = P_TOK_OBJ_END;
							break;

			case '=':		type = P_TOK_LINK;
							break;

			case ';':		type = P_TOK_END;
							break;

			case '/':		type = P_TOK_PATH;
							break;

			case ',':		type = P_TOK_LIST;
							break;
														
			default:		type = P_TOK_UNKPUNCT;		// return that it's a ID
							break;
		}
	}

	// (4) Now advance pointer, if there is no more pending left say so
	// in all cases return result

	if( iscntrl( cur_token[tok_psn] ) )
		tok_psn = NO_PENDING;
		
	// (5) Save unget information
	
	unget_type = type;
	strcpy(unget_val,token);
	
	return( type );
}
