/*
 * Copyright 2002-2008 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.springframework.config.java.model;

import java.lang.annotation.Annotation;
import java.util.ArrayList;
import java.util.List;

import org.springframework.util.Assert;


/** TODO: JAVADOC */
public class ModelMethod {

    private final String name;
    private final int modifiers;
    private final List<Annotation> annotations = new ArrayList<Annotation>();

    /** optionally set. */
    private ModelClass declaringClass;
    private int lineNumber;

    public ModelMethod(String name, int modifiers, Annotation[] annotations) {
        Assert.hasText(name);
        this.name = name;

        Assert.notNull(annotations);
        for(Annotation annotation : annotations)
            this.annotations.add(annotation);

        Assert.isTrue(modifiers >= 0, "modifiers must be non-negative: " + modifiers);
        this.modifiers = modifiers;
    }

    public String getName() {
        return name;
    }

    /**
     * @see  java.lang.reflect.Modifier
     */
    public int getModifiers() {
        return modifiers;
    }

    protected Annotation[] getAnnotations() {
        return annotations.toArray(new Annotation[] { });
    }

    public void addAnnotation(Annotation annotation) {
        annotations.add(annotation);
    }

    public void setDeclaringClass(ModelClass declaringClass) {
        this.declaringClass = declaringClass;
    }

    public ModelClass getDeclaringClass() {
        return declaringClass;
    }

    public void setLineNumber(int lineNumber) {
        this.lineNumber = lineNumber;
    }

    public int getLineNumber() {
        return lineNumber;
    }

    /**
     * Create a ModelMethod representation of a.{@link java.lang.reflect.Method}
     */
    public static ModelMethod forMethod(java.lang.reflect.Method method) {
        ModelMethod modelMethod = new ModelMethod(method.getName(), method.getModifiers(), method.getAnnotations());
        modelMethod.setDeclaringClass(ModelClass.forClass(method.getDeclaringClass()));
        return modelMethod;
    }

    @Override
    public String toString() {
        return String.format("%s: name=%s; modifiers=%d",
                             getClass().getSimpleName(), name, modifiers);
    }

    /* NOTE: bi-directional relationship between ModelMethod and ModelClass means that
     * equals() should not evaluate declaringClass.  It will cause a stack overflow */
    @Override
    public int hashCode() {
        final int prime = 31;
        int result = 1;
        result = prime * result + ((annotations == null) ? 0 : annotations.hashCode());
        result = prime * result + modifiers;
        result = prime * result + ((name == null) ? 0 : name.hashCode());
        return result;
    }

    /* NOTE: bi-directional relationship between ModelMethod and ModelClass means that
     * hashCode() should not evaluate declaringClass.  It will cause a stack overflow */
    @Override
    public boolean equals(Object obj) {
        if (this == obj)
            return true;
        if (obj == null)
            return false;
        if (getClass() != obj.getClass())
            return false;
        ModelMethod other = (ModelMethod) obj;
        if (annotations == null) {
            if (other.annotations != null)
                return false;
        } else if (!annotations.equals(other.annotations))
            return false;
        if (modifiers != other.modifiers)
            return false;
        if (name == null) {
            if (other.name != null)
                return false;
        } else if (!name.equals(other.name))
            return false;
        return true;
    }

}
