###########
##	Cessna 208B with G1000 Electrical System
##	
##		started 07/2023 by Bea Wolf
##

#	References:
#		POH, p.7-71ff.
#					Standby Alternator System Supplement p. S20-3 ff.
#
#		Battery: RG380E-44		https://batterymanagement.concordebattery.com/BatteryDocs/Brochures/RG-380E-44.pdf

#	Property Setup
var elec = props.globals.getNode("/systems/electrical");
var elec_out = elec.getNode("outputs");
var elec_bat = elec.getNode("battery", 1);

var swit_e = props.globals.getNode("/controls/electric", 1);
var swit_f = props.globals.getNode("/controls/flight", 1);
var swit_l = props.globals.getNode("/controls/lighting", 1);

var switches = {
	battery:	swit_e.initNode("battery-switch", 0, "BOOL"),
	generator:	swit_e.initNode("engine[0]/generator", 0, "INT"), # 0 = TRIP, 1 = ON, 2 = RESET
	sby_pwr:	swit_e.initNode("stby-pwr-switch", 0, "BOOL"),
	ext_pwr:	swit_e.initNode("ext-pwr-switch", 0, "BOOL"),
	avionics:
	[
	swit_e.initNode("avionics-pwr-switch1", 0, "BOOL"),
	swit_e.initNode("avionics-pwr-switch2", 0, "BOOL"),
	],
	av_bus_tie:	swit_e.initNode("avionics-bus-tie-switch", 0, "BOOL"),
	av_sby_pwr:	swit_e.initNode("avionics-stby-pwr-switch", 0, "BOOL"),
	
	starter:	swit_e.initNode("starter-switch", 0, "INT"), # 0 = OFF, 1 = START, 2 = MOTOR
	fuel_boost: props.globals.initNode("controls/fuel/boostpump", 0, "INT"), # 0 = OFF, 1 = NORM, 2 = ON
	ignition:	props.globals.initNode("controls/engines/engine[0]/ignition-int", 0, "BOOL"),
	
	cabin_lts:	swit_e.initNode("cabin-lights", 0, "BOOL"),
	
	strobe:	swit_e.initNode("strobe-switch", 0, "BOOL"),
	beacon:	swit_e.initNode("bcn-switch", 0, "BOOL"),
	ldg_light: [
		swit_e.initNode("leftldg-switch", 0, "BOOL"),
		swit_e.initNode("rightldg-switch", 0, "BOOL"),
	],
	taxi_light:	swit_e.initNode("taxi-switch", 0, "BOOL"),
	nav_lights:	swit_e.initNode("nav-switch", 0, "BOOL"),
	
	flt_panel_light: [
		swit_l.initNode("flight-panel[0]", 0.0, "DOUBLE"),
		swit_l.initNode("flight-panel[1]", 0.0, "DOUBLE"),
	],
	flood_light: [
		swit_l.initNode("flood[0]", 0.0, "DOUBLE"),
		swit_l.initNode("flood[1]", 0.0, "DOUBLE"),
	],
	eng_instr_light: swit_l.initNode("engine-instruments", 0.0, "DOUBLE"),
	lwr_panel_ped_ovhd_light: swit_l.initNode("lwr-panel-ped-ovhd", 0.0, "DOUBLE"),
	sw_ckt_bkr_panel_light: swit_l.initNode("sw-ckt-bkr-panel", 0.0, "DOUBLE"),
	radio_panel_light: swit_l.initNode("radio-panel", 0.0, "DOUBLE"),
	
	flap_cmd:	swit_f.getNode("flap-cmd-int", 1),
	flap_sby:	swit_f.getNode("flap-standby-cmd-int", 1),
};

var starter_fdm = props.globals.getNode("controls/engines/engine[0]/starter");

var generator_control = elec_out.initNode("generator-control", 0.0, "DOUBLE");
var generator_field = elec_out.initNode("generator-field", 0.0, "DOUBLE");

var breakers = props.globals.getNode("/controls/circuit-breakers", 1);

var bus_breakers = {
	# Each "sub-bus" has its own master braker (ref. POH 1998, p.7-71)
	bus_1: [
		breakers.initNode("bus-1-pwr[0]", 1, "BOOL"),
		breakers.initNode("bus-1-pwr[1]", 1, "BOOL"),
		breakers.initNode("bus-1-pwr[2]", 1, "BOOL"),
	],
	bus_2: [
		breakers.initNode("bus-2-pwr[0]", 1, "BOOL"),
		breakers.initNode("bus-2-pwr[1]", 1, "BOOL"),
		breakers.initNode("bus-2-pwr[2]", 1, "BOOL"),
	],
	bus_1_stby:	breakers.initNode("bus-1-standby", 1, "BOOL"),
	bus_2_stby: breakers.initNode("bus-2-standby", 1, "BOOL"),
};

var freeze_replay	=	props.globals.getNode("/sim/freeze/replay-state");

var delta_sec	=	props.globals.getNode("sim/time/delta-sec");
var dt = 0.0;

## Lights
#				EXTERIOR (ref. POH 1998, p.7-77ff.)
#	Landing Lights ref. https://flywat.com/collections/featured-products/products/parmetheus%E2%84%A2-g3-par-46-landing-light (LED replacement)
#		LH	systems/electrical/outputs/landing-light[0]
#		RH	systems/electrical/outputs/landing-light[1]
#	Taxi/Recognition Lights ref. https://flywat.com/collections/led-aircraft-landing-taxi-lights/products/parmetheus%E2%84%A2-g3 (LED replacement)
#		LH	systems/electrical/outputs/taxi-light[0]
#		RH	systems/electrical/outputs/taxi-light[1]
#	Navigation Lights (3) ref. https://flywat.com/products/90613-series-led-position-anti-collision-light (LED replacement)
#			systems/electrical/outputs/navigation-lights
#	Strobe Lights ref. https://flywat.com/products/90613-series-led-position-anti-collision-light (LED replacement)
#			systems/electrical/outputs/strobe-lights
#			45 flashes per minute, duration 0.25 sec (ref. https://cdn.shopify.com/s/files/1/0080/5598/0128/files/14455_17a83f0a-128a-4696-9d76-206ba8e08190.pdf?v=1583424557)
#	Flashing Beacon	ref. https://flywat.com/collections/cessna-products/products/orion-360-beacons (LED replacement)
#			systems/electrical/outputs/beacon
#			45 flashes per minute, duration 0.14 sec (ref. https://cdn.shopify.com/s/files/1/0080/5598/0128/files/14455_17a83f0a-128a-4696-9d76-206ba8e08190.pdf?v=1583424557)
#
#				INTERNAL (ref. POH 1998, p, 7-80ff.)
#	Flight Panel Lights														INSTRUMENT LIGHTS
#		L FLT PNL:		Left portion of instr panel, integral lights
#		R FLT PNL:		Right portion of instr panel, (integral lights)
#	Flood Lights															RADIO/FLOOD LIGHT
#		L FLOOD:		Right overhead floodlight, shining on the left panel
#		R FLOOD:		Left overhead floodlight, shining on the right panel
#	LWR PANEL/PED/OVHD (Lower Panel, Pedestal, Overhead)									RADIO/FLOOD LIGHT
#			Post Lights for Lower Panel
#			Floodlights for Lower Panel + Pedestal
#			Overhead Panel + OAT Gauge
#	SW/CKT BKR (Switch/Circuit Breaker Panel)											RADIO/FLOOD LIGHT
#			two floodlights illuminating left side switch and circuit breaker panel
#	ENG INST (Engine Instruments)													INSTRUMENT LIGHTS / ANNUN PANEL (2)
#			Post lights for engine instruments (top of panel), integral annunciator panel lights
#	RADIO (Radio Panel)														RADIO/FLOOD LIGHT
#			Integral Lights and Displays of radio panel instruments
#			Special: extreme counter-clockwise (i.e. "0"): displays to full bright (daytime mode)
#	Control Wheel Maplight														MAP LIGHT
#	Cabin Lights															CABIN LIGHTS (Battery Bus)
#	Passenger Reading Lights													READING LIGHTS
#	No Smoking/Fasten Seatbelt Lights

aircraft.light.new("sim/model/lights/strobe", [0.25, 1.08], switches.strobe);
aircraft.light.new("sim/model/lights/beacon", [0.14, 1.19], switches.beacon);


# Helper functions
var check_or_create = func ( prop, value, type ) {
	var obj = props.globals.getNode(prop, 1);
	if( obj.getValue() == nil ){
		return props.globals.initNode(prop, value, type);
	} else {
		return obj;
	}
}

#	Battery Module
var BatteryClass = {
	# Constructor
	new: func( ideal_volts, ideal_amps, amp_hours, charge_amps, n ){
		var charge_prop = elec_bat.getNode( "charge["~n~"]" );
		var charge = nil;
		if( getprop("/systems/electrical/battery/charge["~n~"]") != nil ){			# If the battery charge has been set from a previous FG instance
			charge = charge_prop.getDoubleValue();
		} else {
			charge = 1.0;
			charge_prop = elec_bat.initNode("charge["~n~"]", 1.0, "DOUBLE");
		}
		var obj = {
			parents: [BatteryClass],
			ideal_volts: ideal_volts,
			ideal_amps: ideal_amps,
			amp_hours: amp_hours,
			charge_amps: charge_amps,
			charge: charge,
			charge_prop: charge_prop,
			n: n 
		};
		return obj;
	},
	# Passing in positive amps means the battery will be discharged.
	# Negative amps indicates a battery charge.
	apply_load: func( amps ){
		var old_charge = me.charge_prop.getDoubleValue();
		if( freeze_replay.getBoolValue() ){
			return me.amp_hours * old_charge;
		}
		var amphrs_used = amps * dt / 3600.0;
		var fraction_used = amphrs_used / me.amp_hours;
		
		var new_charge = std.max(0.0, std.min(old_charge - fraction_used, 1.0));
		
		if (new_charge < 0.1 and old_charge_percent >= 0.1)
			gui.popupTip("Warning: Low battery! Enable alternator or apply external power to recharge battery!", 10);
		me.charge = new_charge;
		me.charge_prop.setDoubleValue( new_charge );
		return me.amp_hours * new_charge;
	},
	# Return output volts based on percent charged.  Currently based on a simple
	# polynomial percent charge vs. volts function.
	get_output_volts: func() {
		var x = 1.0 - me.charge;
		var tmp = -(3.0 * x - 1.0);
		var factor = ( math.pow( tmp, 5) + 32 ) / 32;
		return me.ideal_volts * factor;
	},
	# Return output amps available.  This function is totally wrong and should be
	# fixed at some point with a more sensible function based on charge percent.
	# There is probably some physical limits to the number of instantaneous amps
	# a battery can produce (cold cranking amps?)
	get_output_amps: func() {
		var x = 1.0 - me.charge;
		var tmp = -(3.0 * x - 1.0);
		var factor = ( math.pow( tmp, 5) + 32) / 32;
		return me.ideal_amps * factor;
	},
	# Set the current charge instantly to 100 %.
	reset_to_full_charge: func() {
		me.apply_load(-(1.0 - me.charge) * me.amp_hours, 3600);
	},
	# Get current charge
	get_charge: func() {
		return me.charge;
	}
};

#	Alternator Module (used for "starter/generator" and "standby alternator")

# var alternator = AlternatorClass.new(num,switch,rpm_source,rpm_threshold,volts,amps);
AlternatorClass = {
	new : func (num, src,thr,vlt,amp){
		m = { parents : [AlternatorClass] };
		m.meter =  elec.getNode("gen-load["~num~"]",1);
		m.meter.setDoubleValue(0);
		m.gen_output =  props.globals.getNode("engines/engine[0]/amp-v",1);
		m.gen_output.setDoubleValue(0);
		m.meter.setDoubleValue(0);
		m.rpm_source =  props.globals.getNode(src,1);
		m.rpm_threshold = thr;
		m.ideal_volts = vlt;
		m.ideal_amps = amp;
		return m;
	},
	
	apply_load : func(load) {
		var cur_volt=me.gen_output.getValue();
		var cur_amp=me.meter.getValue();
		if(cur_volt >1){
			var factor=1/cur_volt;
			gout = (load * factor);
			if(gout>1)gout=1;
		}else{
			gout=0;
		}
		if(cur_amp > gout)me.meter.setValue(cur_amp - 0.01);
		if(cur_amp < gout)me.meter.setValue(cur_amp + 0.01);
	},
	
	get_output_volts : func {
		var out = 0;
		var factor = me.rpm_source.getDoubleValue() / me.rpm_threshold;
		if ( factor > 1.0 )factor = 1.0;
		var out = (me.ideal_volts * factor);
		me.gen_output.setValue(out);
		if (out > 1) return out;
		return 0;
	},
	
	get_output_amps : func {
		var factor = me.rpm_source.getValue() / me.rpm_threshold;
		if ( factor > 1.0 ) {
			factor = 1.0;
		}
		var ampout = me.ideal_amps * factor;
		return ampout;
	}
};

#	Define Battery and Generator/Alternators
var battery = BatteryClass.new( 24.0, 100, 42.0, 100, 0 );	# ideal amps and charge amps are guesses

var starter_generator = AlternatorClass.new( 0, "engines/engine[0]/n1", 25.0, 28.0, 100.0 );	# RPM threshold and amps are guesses
var alternator = AlternatorClass.new( 1, "engines/engine[0]/n1", 25.0, 28.0, 75.0 ); # RPM threshold is a guess

# Bus and Consumer Modules
var consumer = {
	new: func( name, switch, load ){
		m = { parents : [consumer] };
		m.cb = breakers.initNode(name, 1, "BOOL");
		m.switch_type = "none";
		if( switch != nil ){
			m.switch = switch;
			if ( switch.getType() == "DOUBLE" ) {
				m.switch_type = "double";
			} else if ( switch.getType() == "BOOL" ) {
				m.switch_type = "bool";
			} else {
				die("Consumer (non-int) switch of unsupported type: "~ switch.getType() ~ "!");
			}
		} else {
			m.switch = nil;
		}
		m.output = elec_out.initNode(name, 0.0, "DOUBLE");
		m.load = load;
		return m;
	},
	power: func( bus_volts ){
		if( me.cb.getBoolValue() and bus_volts != 0.0 ){
			if ( me.switch_type == "none" or ( me.switch_type == "bool" and me.switch.getBoolValue() ) ) {
				me.output.setDoubleValue( bus_volts );
				return me.load;
			} else if ( me.switch_type == "double" ) {
				me.output.setDoubleValue( bus_volts * me.switch.getDoubleValue() );
				return me.load * me.switch.getDoubleValue();
			} else {
				me.output.setDoubleValue( 0.0 );
				return 0.0;
			}
		} else {
			me.output.setDoubleValue( 0.0 );
			return 0.0;
		}
	},
};
# Consumer with support for integer switches
var consumer_int = {
	new: func( name, switch, load, int, mode ){
		m = { parents : [consumer_int] };
		m.cb = breakers.initNode(name, 1, "BOOL");
		if ( switch.getType() == "INT" ) {
			m.switch = switch;
			m.int = int;
			# Mode: 0 means "=="; 1 means "!="
			if( mode != nil ){
				m.mode = mode;
			} else {
				m.mode = 0;
			}
		} else {
			die("Consumer (int) switch of unsupported type: "~ switch.getType() ~ "!");
		}
		m.output = elec_out.initNode(name, 0.0, "DOUBLE");
		m.load = load;
		return m;
	},
	power: func( bus_volts ){
		if( me.cb.getBoolValue() and bus_volts != 0.0 ){
			if ( ( ( me.mode == 0 and me.switch.getIntValue() == me.int ) or ( me.mode == 1 and me.switch.getIntValue() != me.int ) ) ) {
				me.output.setDoubleValue( bus_volts );
				return me.load;
			} else {
				me.output.setDoubleValue( 0.0 );
				return 0.0;
			}
		} else {
			me.output.setDoubleValue( 0.0 );
			return 0.0;
		}
	},
};

var bus = {
	new: func( name, on_update, consumers ) {
		m = { parents : [bus] };
		m.name = name;
		m.volts = check_or_create("systems/electrical/bus/" ~ name ~ "-volts", 0.0, "DOUBLE");
		m.serviceable = check_or_create("systems/electrical/bus/" ~ name ~ "-serviceable", 1, "BOOL");
		m.on_update = on_update;
		m.bus_volts = 0.0;
		m.consumers = consumers;
		return m;
	},
	update_consumers: func () {
		#print("Update consumers of bus "~ me.name);
		load = 0.0;
		foreach( var c; me.consumers ) {
			load += c.power( me.bus_volts );
		}
		return load;
	},
};

var alternator_bus_load = 0.0;

var alternator_bus = bus.new(
	"alternator-bus",
	func{
		if( switches.standby.getBoolValue() ){
			me.bus_volts = alternator.get_output_volts();
		} else {
			me.bus_volts = 0.0;
		}
		
		alternator.apply_load( alternator_bus_load );
		
		me.volts.setDoubleValue( me.bus_volts );
	},
	[],
);


var bus_1 = [
	bus.new( 
		"bus1-1",
		func( bv ){
			var pwr_src = "";
			me.bus_volts = 0.0;
			
			if( bus_breakers.bus_1[0].getBoolValue() ){
				me.bus_volts = bv;
				pwr_src = "power_distribution_bus";
			}
			if( bus_breakers.bus_1_stby.getBoolValue() and alternator_bus.bus_volts > ( me.bus_volts + 0.1 ) ){
				me.bus_volts = alternator_bus.bus_volts;
				pwr_src = "alternator_bus";
			}
			me.volts.setDoubleValue( me.bus_volts );
			
			var load = me.update_consumers();
			
			if( pwr_src == "power_distribution_bus" ){
				return load;
			} else {
				alternator_bus_load += load;
				return 0.0;
			}
		},
		[
			consumer.new( "flap-motor", switches.flap_cmd, 0.1 ),
			consumer_int.new( "standby-flap-motor", switches.flap_sby, 0.1, 0, 1 ),
			consumer.new( "ignition-exciter", switches.ignition, 0.1 ),
			consumer_int.new( "starter-control", switches.starter, 0.1, 0, 1 ),
			consumer.new( "turn-coordinator", nil, 0.1 ),
			consumer.new( "annun-panel", nil, 0.1 ),
			consumer.new( "generator-control", nil, 0.1 ),
			consumer.new( "generator-field", nil, 0.1 ),
		],
	),
	bus.new( 
		"bus1-2",
		func( bv ){	
			var pwr_src = "";
			me.bus_volts = 0.0;
			
			if( bus_breakers.bus_1[1].getBoolValue() ){
				me.bus_volts = bv;
				pwr_src = "power_distribution_bus";
			}
			if( bus_breakers.bus_1_stby.getBoolValue() and alternator_bus.bus_volts > ( me.bus_volts + 0.1 ) ){
				me.bus_volts = alternator_bus.bus_volts;
				pwr_src = "alternator_bus";
			}
			me.volts.setDoubleValue( me.bus_volts );
			
			var load = me.update_consumers();
			
			if( pwr_src == "power_distribution_bus" ){
				return load;
			} else {
				alternator_bus_load += load;
				return 0.0;
			}
		},
		[
			consumer.new( "fire-detector", nil, 0.1 ),
			consumer.new( "fuel-totalizer", nil, 0.1 ),
			consumer.new( "fuel-qty[0]", nil, 0.1 ),
			consumer.new( "oil-temp", nil, 0.1 ),
			consumer.new( "bleed-air-temp", nil, 0.1 ),
			consumer.new( "aft-vent-blower", nil, 0.1 ),
			consumer.new( "pitot-heat[0]", nil, 0.1 ),
			consumer.new( "windshield-anti-ice", nil, 0.1 ),
			consumer.new( "windshield-anti-ice-control", nil, 0.1 ),
		],
	),
	bus.new( 
		"bus1-3",
		func( bv ){	
			var pwr_src = "";
			me.bus_volts = 0.0;
			
			if( bus_breakers.bus_1[2].getBoolValue() ){
				me.bus_volts = bv;
				pwr_src = "power_distribution_bus";
			}
			if( bus_breakers.bus_1_stby.getBoolValue() and alternator_bus.bus_volts > ( me.bus_volts + 0.1 ) ){
				me.bus_volts = alternator_bus.bus_volts;
				pwr_src = "alternator_bus";
			}
			me.volts.setDoubleValue( me.bus_volts );
			
			var load = me.update_consumers();
			
			if( pwr_src == "power_distribution_bus" ){
				return load;
			} else {
				alternator_bus_load += load;
				return 0.0;
			}
		},
		[
			consumer.new( "landing-light[0]", switches.ldg_light[0], 75.6 ), # 2.7 A at 28 VDC
			consumer.new( "strobe-lights", props.globals.getNode("sim/model/lights/strobe/state", 1), 109.2 ), # 3.9 A at 28 VDC during pulse
			consumer.new( "beacon", props.globals.getNode("sim/model/lights/beacon/state", 1), 39.2 ), # 1.4 A at 28 VDC during pulse
			consumer.new( "map-light", nil, 0.1 ),
			consumer.new( "instrument-lights", nil, 0.1 ), # custom power() function, see below
			consumer.new( "wing-ice-detection-light", nil, 0.1 ),
			consumer.new( "DG[0]", nil, 0.1 ),
			consumer.new( "landing-gear-control", nil, 0.1 ),
			consumer.new( "landing-gear-pump", nil, 0.1 ),
		],
	),
];

# INSTRUMENT LIGHTS: This CB/Consumer includes different lights controlled by different knobs
bus_1[2].consumers[4].output = [
	elec_out.initNode("flt-panel-lights-norm[0]", 0.0, "DOUBLE"),
	elec_out.initNode("flt-panel-lights-norm[1]", 0.0, "DOUBLE"),
	elec_out.initNode("eng-instr-lights-norm[0]", 0.0, "DOUBLE"),
];
	

bus_1[2].consumers[4].power = func( bus_volts ){
	
	if( !me.cb.getBoolValue() or bus_volts < 10.0 ) return 0.0;
	
	me.output[0].setDoubleValue( bus_volts / 28 * switches.flt_panel_light[0].getDoubleValue() );
	me.output[1].setDoubleValue( bus_volts / 28 * switches.flt_panel_light[1].getDoubleValue() );
	me.output[2].setDoubleValue( bus_volts / 28 * switches.eng_instr_light.getDoubleValue() );
	
	return me.load * ( switches.flt_panel_light[0].getDoubleValue() + switches.flt_panel_light[1].getDoubleValue() + switches.eng_instr_light.getDoubleValue() ) / 3;
	
}

var bus_2 = [
	bus.new( 
		"bus2-1",
		func( bv ){	
			var pwr_src = "";
			me.bus_volts = 0.0;
			
			if( bus_breakers.bus_2[0].getBoolValue() ){
				me.bus_volts = bv;
				pwr_src = "power_distribution_bus";
			}
			if( bus_breakers.bus_2_stby.getBoolValue() and alternator_bus.bus_volts > ( me.bus_volts + 0.1 ) ){
				me.bus_volts = alternator_bus.bus_volts;
				pwr_src = "alternator_bus";
			}
			me.volts.setDoubleValue( me.bus_volts );
			
			var load = me.update_consumers();
			
			if( pwr_src == "power_distribution_bus" ){
				return load;
			} else {
				alternator_bus_load += load;
				return 0.0;
			}
		},
		[
			consumer.new( "left-ventilation-blower", nil, 0.1 ),
			consumer.new( "right-ventilation-blower", nil, 0.1 ),
			consumer.new( "aux-fuel-pump", nil, 0.1 ),
			consumer.new( "fuel-control-heater", nil, 0.1 ),
			consumer.new( "turn-coordinator[1]", nil, 0.1 ),
			consumer.new( "annun-panel[1]", nil, 0.1 ),
			consumer.new( "stall-warning", nil, 0.1 ),
			consumer.new( "air-conditioner-control", nil, 0.1 ),
		],
	),
	bus.new(
		"bus2-2",
		func( bv ){	
			var pwr_src = "";
			me.bus_volts = 0.0;
			
			if( bus_breakers.bus_2[1].getBoolValue() ){
				me.bus_volts = bv;
				pwr_src = "power_distribution_bus";
			}
			if( bus_breakers.bus_2_stby.getBoolValue() and alternator_bus.bus_volts > ( me.bus_volts + 0.1 ) ){
				me.bus_volts = alternator_bus.bus_volts;
				pwr_src = "alternator_bus";
			}
			me.volts.setDoubleValue( me.bus_volts );
			
			var load = me.update_consumers();
			
			if( pwr_src == "power_distribution_bus" ){
				return load;
			} else {
				alternator_bus_load += load;
				return 0.0;
			}
		},
		[
			consumer.new( "itt", nil, 0.1 ),
			consumer.new( "fuel-flow", nil, 0.1 ),
			consumer.new( "fuel-qty[1]", nil, 0.1 ),
			consumer.new( "airspeed-warning", nil, 0.1 ),
			consumer.new( "prop-overspeed-test", nil, 0.1 ),
			consumer.new( "de-ice-boot", nil, 0.1 ),
			consumer.new( "pitot-heat[1]", nil, 0.1 ),
			consumer.new( "propeller-anti-ice", nil, 0.1 ),
			consumer.new( "propeller-anti-ice-control", nil, 0.1 ),
		],
	),
	bus.new( 
		"bus2-3",
		func( bv ){	
			var pwr_src = "";
			me.bus_volts = 0.0;
			
			if( bus_breakers.bus_2[2].getBoolValue() ){
				me.bus_volts = bv;
				pwr_src = "power_distribution_bus";
			}
			if( bus_breakers.bus_2_stby.getBoolValue() and alternator_bus.bus_volts > ( me.bus_volts + 0.1 ) ){
				me.bus_volts = alternator_bus.bus_volts;
				pwr_src = "alternator_bus";
			}
			me.volts.setDoubleValue( me.bus_volts );
			
			var load = me.update_consumers();
			
			if( pwr_src == "power_distribution_bus" ){
				return load;
			} else {
				alternator_bus_load += load;
				return 0.0;
			}
		},
		[
			consumer.new( "landing-light[1]", switches.ldg_light[1], 75.6 ), # 2.7 A at 28 VDC
			consumer.new( "taxi-lights", switches.taxi_light, 53.2 ), # 1.9 A at 28 VDC
			consumer.new( "navigation-lights", switches.nav_lights, 6.44 ), # 0.23 A at 28 VDC
			consumer.new( "radio-flood-lights", nil, 0.1 ),
			consumer.new( "reading-lights", nil, 0.1 ),
			consumer.new( "attitude-gyro", nil, 0.1 ),
			consumer.new( "seatbelt-sign", nil, 0.1 ),
			consumer.new( "encoding-altimeter", nil, 0.1 ),
			consumer.new( "elevator-trim", nil, 0.1 ),
		],
	),
];


# RADIO/FLOOD LIGHTS: This CB/Consumer includes different lights controlled by different knobs


#	Flood Lights															RADIO/FLOOD LIGHT
#		L FLOOD:		Right overhead floodlight, shining on the left panel
#		R FLOOD:		Left overhead floodlight, shining on the right panel
#	LWR PANEL/PED/OVHD (Lower Panel, Pedestal, Overhead)									RADIO/FLOOD LIGHT
#			Post Lights for Lower Panel
#			Floodlights for Lower Panel + Pedestal
#			Overhead Panel + OAT Gauge
#	SW/CKT BKR (Switch/Circuit Breaker Panel)											RADIO/FLOOD LIGHT
#			two floodlights illuminating left side switch and circuit breaker panel
#	RADIO (Radio Panel)														RADIO/FLOOD LIGHT
bus_2[2].consumers[3].output = [
	elec_out.initNode("flood-light-norm[0]", 0.0, "DOUBLE"),
	elec_out.initNode("flood-light-norm[1]", 0.0, "DOUBLE"),
	elec_out.initNode("lwr-panel-ped-ovhd-lights-norm", 0.0, "DOUBLE"),
	elec_out.initNode("sw-ckt-bkr-panel-lights-norm", 0.0, "DOUBLE"),
	elec_out.initNode("radio-panel-lights-norm", 0.0, "DOUBLE"),
];
bus_2[2].consumers[3].switch = [
	switches.flood_light[0],
	switches.flood_light[1],
	switches.lwr_panel_ped_ovhd_light,
	switches.sw_ckt_bkr_panel_light,
	switches.radio_panel_light,
];

bus_2[2].consumers[3].power = func( bus_volts ){
	
	if( !me.cb.getBoolValue() or bus_volts < 10.0 ) return 0.0;
	
	var act_load = 0.0;
	forindex( var i; me.switch ){
		me.output[ i ].setDoubleValue( bus_volts / 28 * me.switch[ i ].getDoubleValue() );
		act_load += me.switch[ i ].getDoubleValue();
	}
	return act_load / 5 * me.load;
	
}

var avionics_bus_1_load = 0.0;

var avionics_bus = [
	bus.new(
		"avionics-bus1",
		func( bv ){	
			var pwr_src = "";
			if( switches.avionics[0].getBoolValue() ){
				me.bus_volts = bv;
				pwr_src = "power_distribution_bus";
			}
			if( switches.av_sby_pwr.getBoolValue() and alternator_bus.bus_volts > ( me.bus_volts + 0.1 ) ){
				me.bus_volts = alternator_bus.bus_volts;
				pwr_src = "alternator_bus";
			}
				
			me.volts.setDoubleValue( me.bus_volts );
			
			var load = me.update_consumers();
			
			load += avionics_bus_1_load;
			
			if( pwr_src == "power_distribution_bus" ){
				return load;
			} else {
				alternator_bus_load += load;
				return 0.0;
			}
		},
		[
			consumer.new( "comm[0]", nil, 0.1 ),
			consumer.new( "nav[0]", nil, 0.1 ),
			consumer.new( "engine-interface", nil, 0.1 ),
			consumer.new( "adc[0]", nil, 0.1 ),
			consumer.new( "ahrs[0]", nil, 0.1 ),
			consumer.new( "pfd[0]", nil, 0.1 ),
			consumer.new( "transponder[0]", nil, 0.1 ),
			consumer.new( "adf[0]", nil, 0.1 ),
		],
	),
	bus.new(
		"avionics-bus2",
		func( bv ){	
			var pwr_src = "";
			if( switches.avionics[1].getBoolValue() ){
				me.bus_volts = bv;
				pwr_src = "power_distribution_bus";
			}
			if( switches.av_bus_tie.getBoolValue() and avionics_bus[0].bus_volts > ( me.bus_volts + 0.1 ) ){
				me.bus_volts = avionics_bus[0].bus_volts;
				pwr_src = "avionics-bus1";
			}
				
			me.volts.setDoubleValue( me.bus_volts );
			
			var load = me.update_consumers();
			
			if( pwr_src == "power_distribution_bus" ){
				return load;
			} else {
				avionics_bus_1_load += load;
				return 0.0;
			}
		},
		[
			consumer.new( "comm[1]", nil, 0.1 ),
			consumer.new( "nav[1]", nil, 0.1 ),
			consumer.new( "mfd", nil, 0.1 ),
			consumer.new( "adc[1]", nil, 0.1 ),
			consumer.new( "ahrs[1]", nil, 0.1 ),
			consumer.new( "pfd[1]", nil, 0.1 ),
			consumer.new( "transponder[1]", nil, 0.1 ),
			consumer.new( "dme", nil, 0.1 ),
			consumer.new( "tas", nil, 0.1 ),
			consumer.new( "xm-dta-link", nil, 0.1 ),
			consumer.new( "hf-transmitter", nil, 0.1 ),
			consumer.new( "hf-receiver", nil, 0.1 ),
			consumer.new( "radio-altimeter", nil, 0.1 ),
			consumer.new( "radar", nil, 0.1 ),
		],
	),
];

var power_distribution_bus = bus.new(
	"power-distribution-bus",
	func( bv ){
		me.bus_volts = 0.0;
		var pwr_src = "";
		var load = 0.0;
		if( me.serviceable.getBoolValue() ){
			if( switches.battery.getBoolValue() ){
				me.bus_volts = bv;
				pwr_src = "bat";
			}
			if( switches.generator.getIntValue() == 1 and generator_control.getDoubleValue() > 10.0 and generator_field.getDoubleValue() > 10.0 and starter_generator.get_output_volts() > me.bus_volts ){
				me.bus_volts = starter_generator.get_output_volts();
				pwr_src = "gen";
			}	
		}
		load += me.update_consumers();
		foreach( var el; bus_1 ){
			load += el.on_update( me.bus_volts );
		}
		foreach( var el; bus_2 ){
			load += el.on_update( me.bus_volts );
		}
		foreach( var el; avionics_bus ){
			load += el.on_update( me.bus_volts );
		}
		
		me.volts.setDoubleValue( me.bus_volts );
		
		if( pwr_src == "gen" ){
			load += battery.charge_amps;
			starter_generator.apply_load( load );
			return -battery.charge_amps;
		} elsif( pwr_src == "bat" ){
			return load;
		} else {
			return 0.0;
		}
	},
	[],
);


# Battery Bus - continuously energized
var battery_bus = bus.new(
	"battery-bus",
	func{
		var load = 0.0;
		if( me.serviceable.getBoolValue() ){
			me.bus_volts = battery.get_output_volts();			
		} else {
			me.bus_volts = 0.0;
		}
		load += me.update_consumers();
		load += power_distribution_bus.on_update( me.bus_volts );
		
		me.volts.setDoubleValue( me.bus_volts );
		
		battery.apply_load( load );
	},
	[
		consumer.new( "clock", nil, 0.1 ),
		consumer.new( "cabin-lights", switches.cabin_lts, 0.1 ),
	],
);

var flap_power = props.globals.getNode("systems/electrical/outputs/flap-motor");
var stby_flap_power = props.globals.getNode("systems/electrical/outputs/standby-flap-motor");
var cmd_flaps = props.globals.getNode("controls/flight/flaps");
var int_flaps = props.globals.getNode("controls/flight/flaps-int");
var flap_speed = 0.25; # full traverse takes 4 seconds, complete guess

var clock_volts = props.globals.getNode("systems/electrical/outputs/clock");
var clock_powered = props.globals.initNode("/instrumentation/clock/powered", 0, "BOOL");

var update_electrical = func {
	dt = delta_sec.getDoubleValue();
	
	alternator_bus_load = 0.0;
	avionics_bus_1_load = 0.0;
	
	battery_bus.on_update();
	
	if( bus_1[0].consumers[3].output.getDoubleValue() > 10.0 ){
		starter_fdm.setBoolValue( 1 );
	} else {
		starter_fdm.setBoolValue( 0 );
	}
	
	var flap_c = cmd_flaps.getDoubleValue();
	var flap_i = int_flaps.getDoubleValue();
	# Normal Flap Operation
	if( flap_power.getDoubleValue() > 15 ){
		flap_i += math.clamp( flap_c - flap_i, -flap_speed * dt, flap_speed * dt);
		int_flaps.setDoubleValue( flap_i );
	} elsif( stby_flap_power.getDoubleValue() > 15 ){
		flap_i += math.clamp( switches.flap_sby.getIntValue() * flap_speed * dt, 0, 1 ); # the standby flap motor has no limiting mechanism, but we don't simulate breaking the motor yet
		int_flaps.setDoubleValue( flap_i );
	}
	
	if( clock_volts.getDoubleValue() > 9 ) clock_powered.setBoolValue( 1 );
	else clock_powered.setBoolValue( 0 );
}

var electrical_updater = maketimer( 0.0, update_electrical );
electrical_updater.simulatedTime = 1;
electrical_updater.start();
