##
# DG LS8 (sc) Electrical System
#	by Benedikt Wolf (D-ECHO) 05/2020

#	adapted from: Cessna 172p Electrical System


#	Additional References
#		FES.1	FES-LAK-M100 Engine Manual (similar to LS8 installation)	https://front-electric-sustainer.com/wp-content/uploads/2022/01/FES-LAK-M100-MOTOR-manual-v1.51.pdf
#		FES.2	FES-LS8-P1-102 Propeller Manual					https://front-electric-sustainer.com/wp-content/uploads/2022/01/FES-LS8-P1-102-PROPELLER-manual-v1.23.pdf
#		FES.3	FES Battery Gen2 Pack Manual					https://front-electric-sustainer.com/wp-content/uploads/2022/01/FES-BATTERY-PACK-GEN2-14S-40Ah-manual-v1.27.pdf
#		FES.4	FES FCU Instrument Manual					https://front-electric-sustainer.com/wp-content/uploads/2022/01/FES-FCU-INSTRUMENT-manual-v1.93.pdf
#		FES.5	FES Flight Manual						https://front-electric-sustainer.com/wp-content/uploads/2020/05/FES-Flight-manual-v1.18.pdf
#		FES.6	FES Maintenance Manual						https://front-electric-sustainer.com/wp-content/uploads/2020/05/FES-Maintenance-manual-v1.16.pdf
#		FES.7	FES System Manual						https://front-electric-sustainer.com/wp-content/uploads/2020/05/FES-System-manual-v1.25.pdf

var volts = 0.0;

var bus_volts = 0.0;

var ammeter_ave = 0.0;

#	Property Setup
var electrical	= props.globals.initNode("/systems/electrical");
var serviceable	= electrical.initNode("serviceable", 1, "BOOL");
var batt_prop	= electrical.initNode("battery");
var output_prop	= electrical.initNode("outputs");
var cb_prop	= props.globals.initNode("/controls/circuit-breakers");
var FES_p	= props.globals.initNode("/systems/FES");

var volts	=	electrical.initNode("volts", 0.0, "DOUBLE");
var amps	=	electrical.initNode("amps",  0.0, "DOUBLE");
var switches = {
	master:	props.globals.getNode("/controls/electric/battery-switch"),
	xpdr:	props.globals.getNode("/controls/electric/xpdr"),
	radio:	props.globals.getNode("/controls/electric/radio"),
	vario:	props.globals.getNode("/controls/electric/vario"),
	fcu:	props.globals.initNode("/systems/FES/fcu-switch", 0, "BOOL"),
	fes_master:	FES_p.initNode("master-switch", 1, "BOOL"),
};

var radio	=	output_prop.initNode("comm[0]", 0.0, "DOUBLE");
var flarm	=	output_prop.initNode("flarm", 0.0, "DOUBLE");
var s3		=	output_prop.initNode("S3", 0.0, "DOUBLE");
var xpdr	=	output_prop.initNode("transponder", 0.0, "DOUBLE");
var acd57	=	output_prop.initNode("acd-57", 0.0, "DOUBLE");
var fcu		=	output_prop.initNode("fcu", 0.0, "DOUBLE");

var comm_ptt	=	props.globals.getNode("/instrumentation/comm[0]/ptt");
var xpdr_mode	=	props.globals.initNode("/instrumentation/transponder/inputs/knob-mode", 0, "INT");
var xpdr_brt	=	props.globals.getNode("/instrumentation/transponder/brightness", 1);


var serviceable		= FES_p.initNode("serviceable", 1, "BOOL");

var amps		= FES_p.initNode("amps", 0.0, "DOUBLE");
var volts		= FES_p.initNode("volts", 0.0, "DOUBLE");

var engine_volts	= FES_p.initNode("engine-volts", 0.0, "DOUBLE");

var engine_power	= props.globals.getNode("/fdm/jsbsim/propulsion/engine[0]/electrical-power-hp");

var delta_sec		= props.globals.getNode("/sim/time/delta-sec");

# Array of circuit breakers
var circuit_breakers = {
	master: cb_prop.initNode("master", 1, "BOOL"),
};

var freeze_replay	=	props.globals.getNode("/sim/freeze/replay-state");

##
# Battery model class.
#

var BatteryClass = {
	# Constructor
	new: func( ideal_volts, ideal_amps, amp_hours, max_amps, charge_amps, n ){
		var base_prop = electrical.initNode( "battery["~n~"]" );
		var charge_prop = base_prop.getNode("charge", 1);
		var charge	= nil;
		if( getprop("/systems/electrical/battery["~n~"]/charge") != nil ){			# If the battery charge has been set from a previous FG instance
			charge = charge_prop.getDoubleValue();
		} else {
			charge = 1.0;
			charge_prop = base_prop.initNode("charge", 1.0, "DOUBLE");
		}
		var obj = {
			parents: [BatteryClass],
			base_prop:	base_prop,
			ideal_volts:	ideal_volts,
			ideal_amps:	ideal_amps,
			max_amps:	max_amps,
			amp_hours:	amp_hours,
			charge:		charge,
			charge_prop:	charge_prop,
			charge_amps:	charge_amps,
			n: 		n,
			serviceable:	base_prop.initNode("serviceable", 1, "BOOL"),
		};
		return obj;
	},
	# Passing in positive amps means the battery will be discharged.
	# Negative amps indicates a battery charge.
	apply_load: func( amps, dt ){
		if( serviceable.getBoolValue() ){
			if( amps > me.max_amps ){
				serviceable.setBoolValue( 0 );
				return 0;
			}
			var old_charge = me.charge_prop.getDoubleValue();
			if( freeze_replay.getBoolValue() ){
				return me.amp_hours * old_charge;
			}
			var amphrs_used = amps * dt / 3600.0;
			var fraction_used = amphrs_used / me.amp_hours;
			
			var new_charge = std.max(0.0, std.min(old_charge - fraction_used, 1.0));
			
			if (new_charge < 0.1 and old_charge_percent >= 0.1)
				gui.popupTip("Warning: Low battery! Enable alternator or apply external power to recharge battery!", 10);
			me.charge = new_charge;
			me.charge_prop.setDoubleValue( new_charge );
			return me.amp_hours * new_charge;
		} else {
			return 0;
		}
	},
	# Return output volts based on percent charged.  Currently based on a simple
	# polynomial percent charge vs. volts function.
	get_output_volts: func() {
		if( serviceable.getBoolValue() ){
			var x = 1.0 - me.charge;
			var tmp = -(3.0 * x - 1.0);
			var factor = ( math.pow( tmp, 5) + 32 ) / 32;
			return me.ideal_volts * factor;
		} else {
			return 0;
		}
	},
	# Set the current charge instantly to 100 %.
	reset_to_full_charge: func() {
		me.apply_load(-(1.0 - me.charge) * me.amp_hours, 3600);
	},
	# Get current charge
	get_charge: func() {
		return me.charge;
	}
};

############################
####	Battery Packs	####
############################

#	FES Batteries: (ref. 4, p. 5 )
#		2 x FES GEN2 14S Battery
#			(ref. FES.3, p.4):	14 * 40 Ah = 560 Ah, 2.1 kWh, 42 V - 58.3 V, max. 250 A, 9 A,18 A or 30 A charging current

#	12V Battery
##		example glider battery: https://shop.segelflugbedarf24.de/Flugzeugausstattung/Akkus-Energieversorgung/Sonnenschein-Dryfit-A512-12V/6-5-Ah::731.html
##			http://www.sonnenschein.org/A500.htm	(A512-6.5S)
##			ideal volts: 12.0
##			ideal amps: 0.325 (max. 80 / 300 for 5 sec))
##			amp hours: 6.5
##			charge amps: 25
#			max. 3A (fuse): ref. FES.6, p.11

var batteries = [
	BatteryClass.new( 58.3, 200, 128.0, 250.0, 9.0, 0),
	BatteryClass.new( 58.3, 200, 128.0, 250.0, 9.0, 1),
	BatteryClass.new( 12.0, 0.325, 6.5, 3.0, 25, 2),
];

var reset_battery = func {
	foreach( var el; batteries ){
		el.reset_to_full_charge();
	}
}
var reset_circuit_breakers = func {
	# Reset circuit breakers
	foreach(var cb; keys(circuit_breakers)){
		circuit_breakers[cb].setBoolValue( 1 );
	}
}

##
# This is the main electrical system update function.
#

var ElectricalSystemUpdater = {
	new : func {
		var m = {
			parents: [ElectricalSystemUpdater]
		};
		# Request that the update function be called each frame
		m.loop = updateloop.UpdateLoop.new(components: [m], update_period: 0.0, enable: 0);
		return m;
	},
	
	enable: func {
		me.loop.reset();
		me.loop.enable();
	},
	
	disable: func {
		me.loop.disable();
	},
	
	reset: func {
		# Do nothing
	},
	
	update: func (dt) {
		main_battery_bus(dt);
	}
};


var main_battery_bus = func (dt) {
	#	Directly connected to the two high-voltage battery packs via 325A fuse
	var load = 0.0;
	var bus_volts = 0.0;
	
	if( circuit_breakers.master.getBoolValue() and serviceable.getBoolValue() ){
		bus_volts = batteries[0].get_output_volts() + batteries[1].get_output_volts();
	}
	
	# switch state
	load += low_voltage_bus( bus_volts, dt );
	load += engine_bus( bus_volts, dt );
	
	if ( load > 325 ) {
		circuit_breakers.master.setBoolValue( 0 );
	} else {
		batteries[0].apply_load( load / 2, dt );
		batteries[1].apply_load( load / 2, dt );
	}
	
	amps.setDoubleValue( load );
	volts.setDoubleValue( bus_volts );
}

var engine_bus = func( bv, dt ) {
	#	Connected to main battery bus via Main Contactor
	var bus_volts = 0.0;
	if( switches.fes_master.getBoolValue() ){
		bus_volts = bv;
	}
	var load = 0.0;
	
	if( bus_volts > 0 ){
		engine_volts.setDoubleValue( bus_volts );
		
		#Load is engine power divided by volts
		var eng_power = engine_power.getDoubleValue() * 745.7; #hp to watts
		load += eng_power / bus_volts;
	}
	
	# return cumulative load
	return load;
}

var low_voltage_bus = func( bv, dt ){
	#	Connected to main battery bus via DC/DC converter
	#	DC/DC connector efficiency: 69%-89% (https://www.xppower.com/product/DNR60-Series)
	
	var load = 0.0;
	var bus_volts = bv * 0.22; # ca. 12.8 V at max. main battery voltage
	var src = "converter";
	
	if( batteries[2].get_output_volts() > ( bus_volts + 0.5 ) ){
		bus_volts = batteries[2].get_output_volts();
		src = "batt12v";
	}
	
	load += instrument_bus( bus_volts );
	
	if( src == "converter" ){
		if( batteries[2].charge < 0.99 ){
			load += batteries[2].charge_amps;
			batteries[2].apply_load( batteries[2].charge_amps, dt );
		}
		return load / 0.22 / 0.75;
	} else {
		batteries[2].apply_load( load, dt );
		return 0.0;
	}
}

#Load sources:
#	com:		https://www.becker-avionics.com/wp-content/uploads/00_Literature/62Series_DS_en.pdf, p.2
#	vario:		http://www.lx-avionik.de/produkte/s3/
#	xpdr:		https://www.becker-avionics.com/wp-content/uploads/2018/06/BXP6401_IO_issue05.pdf, p.16
#	flarm:		http://flarm.com/wp-content/uploads/man/FLARM_InstallationManual_D.pdf
#	ACD-57:		https://www.air-avionics.com/?page_id=651&lang=de#, p.2

var instrument_bus = func( bv ) {
	#	Connected to low voltage bus via master switch
	var load = 0.0;
	var bus_volts = 0.0;
	if( switches.master.getBoolValue() ){
		bus_volts = bv;
	}
	
	# Electrical Consumers (Instruments)
	if( bus_volts > 9  and switches.radio.getBoolValue() ){	# minimal working voltage
		# Radio
		radio.setDoubleValue( bus_volts );
		if( comm_ptt.getBoolValue() ){
			load += 18 / bus_volts; # transmitting: <= 2 A at 12V
		}else{
			load += 1.56 / bus_volts; # standby: <= 140 mA at 12V
		}
		
	} else {
		radio.setDoubleValue( 0.0 );
	}
	if( bus_volts > 9  and switches.vario.getBoolValue() ){	# minimal working voltage
		# Vario
		s3.setDoubleValue( bus_volts );
		load += 1.380 / bus_volts; # 95mA - 135mA depending on display brightness setting at 12V
		
	} else {
		s3.setDoubleValue( 0.0 );
	}
	
	if( bus_volts > 10 and switches.xpdr.getBoolValue() ){	# minimal working voltage
		# XPDR Power
		if( xpdr_mode.getIntValue() != 0 and bus_volts != 0 ){
			xpdr.setDoubleValue( bus_volts );
			if( xpdr_mode.getIntValue() > 1 ){
				load += ( 0.37 + 0.03 * xpdr_brt.getDoubleValue() ) * 14.0 / bus_volts;	# 0.37 - 0.40 A at 14VDC depending on display brightness setting
			} else {
				load += ( 0.22 + 0.06 * xpdr_brt.getDoubleValue() ) * 14.0 / bus_volts;	# 0.22 - 0.28 A at 14VDC depending on display brightness setting
			}
		} else {
			xpdr.setDoubleValue( 0.0 );
		}
	} else {
		xpdr.setDoubleValue( 0.0 );
	}
	
	if( bus_volts > 8 and switches.xpdr.getBoolValue() ){	# minimal working voltage
		# XPDR Power
		acd57.setDoubleValue( bus_volts );
		load += 0.966 / bus_volts; # 0.07 A at 13.8 V
	} else {
		acd57.setDoubleValue( 0.0 );
	}
	
	# FLARM
	if( bus_volts > 8 ){	# minimal working voltage
		flarm.setDoubleValue( bus_volts );
		load += 0.66 / bus_volts; #FLARM
	} else {
		flarm.setDoubleValue( 0.0 );
	}
	
	# FCU
	if( bus_volts > 8 and switches.fcu.getBoolValue() ){	# TODO minimal working voltage
		fcu.setDoubleValue( bus_volts );
		load += 0.66 / bus_volts; # TODO
	} else {
		fcu.setDoubleValue( 0.0 );
	}
	
	return load;
}


##
# Initialize the electrical system
#

var system_updater = ElectricalSystemUpdater.new();

setlistener("/sim/signals/fdm-initialized", func {
	reset_circuit_breakers();
	
	system_updater.enable();
	print("Electrical System initialized");
});


#	Propeller
var rpm = props.globals.getNode("/engines/engine[0]/rpm", 1);
var blade_angle = props.globals.getNode("/fdm/jsbsim/propulsion/engine[0]/blade-angle", 1);
var prop_pos = props.globals.initNode("/engines/engine[0]/propeller-pos-deg", 0.0, "DOUBLE");	# 0 deg - 360 deg
var auto_pos_speed = 360; # deg per second
var auto_pos_target = -1; # -1 = off, 0-360 target deg

var propeller_loop = func {
	if( rpm.getDoubleValue() > 0.1 ){
		prop_pos.setDoubleValue( math.periodic( 0, 360, prop_pos.getDoubleValue() + rpm.getDoubleValue() * delta_sec.getDoubleValue() * 6 ) );
			#	/ 60 (minute -> second); * 360 (revolutions -> degrees)
	} elsif( auto_pos_target != -1 and prop_pos.getDoubleValue() < auto_pos_target ){
		prop_pos.setDoubleValue( prop_pos.getDoubleValue() + auto_pos_speed * delta_sec.getDoubleValue() );
	} elsif( auto_pos_target == -1 and blade_angle.getDoubleValue() < 85 ){
		auto_pos();
	}
}

var propeller_timer = maketimer( 0.0, propeller_loop );
propeller_timer.simulatedTime = 1;
propeller_timer.start();

# Automatic positioning
var auto_pos = func {
	if( prop_pos.getDoubleValue() < 330 ) {
		auto_pos_target = prop_pos.getDoubleValue() + 30;
		settimer( auto_pos, 1 );
	} else {
		auto_pos_target = 360;
	}
}
