var clamp = func(value,min=0.0,max=0.0){
	if(value < min) {value = min;}
	if(value > max) {value = max;}
	return value;
}

var set_defaults = func {
	setprop("/payload/weight[0]/weight-lb", getprop("/sim/asw20/defaults/pilot-lbs"));
	setprop("/payload/weight[1]/weight-lb", getprop("/sim/asw20/defaults/baggage-lbs"));
	setprop("/payload/weight[2]/weight-lb", getprop("/sim/asw20/defaults/trim-disc-lbs"));
	setprop("/payload/weight[3]/weight-lb", getprop("/sim/asw20/defaults/perm-trim-lbs"));
	setprop("/payload/weight[4]/weight-lb", getprop("/sim/asw20/defaults/instr-pnl-lbs"));
	setprop("/fdm/jsbsim/inertia/waterballast_total-lbs", getprop("/sim/asw20/defaults/water-ballast-lbs"));
}

var weightBalanceFontMapper =  func(family, weight){
	if (family =="'Liberation Mono'"){
		if (weight == "bold"){
			return "LiberationFonts/LiberationMono-Bold.ttf";
		}elsif(weight == "normal"){
			return "LiberationFonts/LiberationMono-Regular.ttf";
		}
	}elsif( family == "'Liberation Sans'" ){
		if (weight == "bold"){
			return "LiberationFonts/LiberationSans-Bold.ttf";
		}elsif(weight == "normal"){
			return "LiberationFonts/LiberationSans-Regular.ttf";
		}
	}elsif( family == "vSHandprinted" ){
		return "vSHandprinted.ttf";
	} else {
		print( "No matching Font found: " );
		debug.dump( caller(1) );
		print( family );
		print( weight );
		return "LiberationFonts/LiberationSans-Regular.ttf";
	}
}

var MyWindow = {
	# Constructor
	#
	# @param size ([width, height])
	new: func(size, type = nil, id = nil)
	{
		var ghost = canvas._newWindowGhost(id);
		var m = {
			parents: [MyWindow, canvas.PropertyElement, ghost],
			_node: props.wrapNode(ghost._node_ghost)
		};
		
		m.setInt("size[0]", size[0]);
		m.setInt("size[1]", size[1]);
		
		# TODO better default position
		m.move(0,0);
		
		# arg = [child, listener_node, mode, is_child_event]
		setlistener(m._node, func m._propCallback(arg[0], arg[2]), 0, 2);
		if( type )
			m.set("type", type);
		
		m._isOpen = 1;
		return m;
	},
	# Destructor
	del: func
	{
		me._node.remove();
		me._node = nil;
		
		if( me["_canvas"] != nil )
		{
			me._canvas.del();
			me._canvas = nil;
		}
		me._isOpen = 0;
	},
	# Create the canvas to be used for this Window
	#
	# @return The new canvas
	createCanvas: func()
	{
		var size = [
		me.get("size[0]"),
		me.get("size[1]")
		];
		
		me._canvas = canvas.new({
			size: [2 * size[0], 2 * size[1]],
			view: size,
			placement: {
				type: "window",
			  id: me.get("id")
			}
		});
		
		me._canvas.addEventListener("mousedown", func me.raise());
		return me._canvas;
	},
	# Set an existing canvas to be used for this Window
	setCanvas: func(canvas_)
	{
		if( !isa(canvas_, canvas.Canvas) )
			return debug.warn("Not a canvas.Canvas");
		
		canvas_.addPlacement({type: "window", index: me._node.getIndex()});
		me['_canvas'] = canvas_;
	},
	# Get the displayed canvas
	getCanvas: func()
	{
		return me['_canvas'];
	},
	getCanvasDecoration: func()
	{
		return canvas.wrapCanvas(me._getCanvasDecoration());
	},
	setPosition: func(x, y)
	{
		me.setInt("tf/t[0]", x);
		me.setInt("tf/t[1]", y);
	},
	move: func(x, y)
	{
		me.setInt("tf/t[0]", me.get("tf/t[0]", 10) + x);
		me.setInt("tf/t[1]", me.get("tf/t[1]", 30) + y);
	},
	# Raise to top of window stack
	raise: func()
	{
		# on writing the z-index the window always is moved to the top of all other
		# windows with the same z-index.
		me.setInt("z-index", me.get("z-index", 0));
	},
	# private:
	_propCallback: func(child, mode)
	{
		if( !me._node.equals(child.getParent()) )
			return;
		var name = child.getName();
		
		# support for CSS like position: absolute; with right and/or bottom margin
		if( name == "right" )
			me._handlePositionAbsolute(child, mode, name, 0);
		else if( name == "bottom" )
			me._handlePositionAbsolute(child, mode, name, 1);
		
		# update decoration on type change
		else if( name == "type" )
		{
			if( mode == 0 )
				settimer(func me._updateDecoration(), 0);
		}
	},
	_handlePositionAbsolute: func(child, mode, name, index)
	{
		# mode
		#   -1 child removed
		#    0 value changed
		#    1 child added
		
		if( mode == 0 )
			me._updatePos(index, name);
		else if( mode == 1 )
			me["_listener_" ~ name] = [
			setlistener
			(
				"/sim/gui/canvas/size[" ~ index ~ "]",
    func me._updatePos(index, name)
			),
			setlistener
			(
				me._node.getNode("size[" ~ index ~ "]"),
			 func me._updatePos(index, name)
			)
			];
			else if( mode == -1 )
				for(var i = 0; i < 2; i += 1)
					removelistener(me["_listener_" ~ name][i]);
	},
	_updatePos: func(index, name)
	{
		me.setInt(	"tf/t[" ~ index ~ "]",  getprop("/sim/gui/canvas/size[" ~ index ~ "]") - me.get(name)  - me.get("size[" ~ index ~ "]"));
	},
	_onClose : func(){
		me.del();
	},
	_updateDecoration: func()
	{
		var border_radius = 9;
		me.set("decoration-border", "25 1 1");
		me.set("shadow-inset", int((1 - math.cos(45 * D2R)) * border_radius + 0.5));
		me.set("shadow-radius", 5);
		me.setBool("update", 1);
		
		var canvas_deco = me.getCanvasDecoration();
		canvas_deco.addEventListener("mousedown", func me.raise());
		canvas_deco.set("blend-source-rgb", "src-alpha");
		canvas_deco.set("blend-destination-rgb", "one-minus-src-alpha");
		canvas_deco.set("blend-source-alpha", "one");
		canvas_deco.set("blend-destination-alpha", "one");
		
		var group_deco = canvas_deco.getGroup("decoration");
		var title_bar = group_deco.createChild("group", "title_bar");
		title_bar
		.rect( 0, 0,
		       me.get("size[0]"),
		       me.get("size[1]"), #25,
		       {"border-top-radius": border_radius} )
		.setColorFill(0.25,0.24,0.22)
		.setStrokeLineWidth(0);
		
		var style_dir = "gui/styles/AmbianceClassic/decoration/";
		
		# close icon
		var x = 10;
		var y = 3;
		var w = 19;
		var h = 19;
		var ico = title_bar.createChild("image", "icon-close")
		.set("file", style_dir ~ "close_focused_normal.png")
		.setTranslation(x,y);
		ico.addEventListener("click", func me._onClose());
		ico.addEventListener("mouseover", func ico.set("file", style_dir ~ "close_focused_prelight.png"));
		ico.addEventListener("mousedown", func ico.set("file", style_dir ~ "close_focused_pressed.png"));
		ico.addEventListener("mouseout",  func ico.set("file", style_dir ~ "close_focused_normal.png"));
		
		title_bar.addEventListener("drag", func(e) {
			if( !ico.equals(e.target) )
				me.move(e.deltaX, e.deltaY);
		});
	}
};

var COLOR = {};
COLOR["Red"] 			= "rgb(244,28,33)";
COLOR["Black"] 			= "#000000";


var SvgWidget = {
	new: func(dialog,canvasGroup,name){
		var m = {parents:[SvgWidget]};
		m._class = "SvgWidget";
		m._dialog 	= dialog;
		m._listeners 	= [];
		m._name 	= name;
		m._group	= canvasGroup;
		return m;
	},
	removeListeners  :func(){
		foreach(l;me._listeners){
			removelistener(l);
		}
		me._listeners = [];
	},
	setListeners : func(instance) {
		
	},
	init : func(instance=me){
		
	},
	deinit : func(){
		me.removeListeners();	
	},
	
};

var WeightWidget = {
	new: func(dialog,canvasGroup,name,widgets){
		var m = {parents:[WeightWidget,SvgWidget.new(dialog,canvasGroup,name)]};
		m._class = "WeightWidget";
		m._widget = {};
		
		foreach(w;keys(widgets)){
			if(widgets[w] != nil){
				if(widgets[w]._class == "PayloadWidget"){
					m._widget[w] = widgets[w];
				}
			}
		}
		
		m._cWeightGrossKg 	= m._group.getElementById("Total_Kg");
		m._cWeightGrossLb 	= m._group.getElementById("Total_Lb");
		m._cWeightNonLiftKg 	= m._group.getElementById("NonLift_Kg");
		m._cWeightNonLiftLb	= m._group.getElementById("NonLift_Lb");
		
		m._cCenterGravityMM	= m._group.getElementById("CG_mm");
		m._cCenterGravityIN	= m._group.getElementById("CG_in");
		m._cEmptyCGMM		= m._group.getElementById("EmptyCG_mm");
		m._cEmptyCGIN		= m._group.getElementById("EmptyCG_in");
		
		m._cWingLoadLbSqFt	= m._group.getElementById("WingLoading_lb-sqft");
		m._cWingLoadKgSqM		= m._group.getElementById("WingLoading_kg-sqm");
		
		m._nCGx	= props.globals.getNode("/fdm/jsbsim/inertia/cg_mass_and_balance-in");
		m._nEmptyCG	= props.globals.getNode("/fdm/jsbsim/inertia/cg_mass_and_balance_empty-in");
		m._nGross 	= props.globals.getNode("/fdm/jsbsim/inertia/weight-lbs");
		m._nNonLift	= props.globals.getNode("/fdm/jsbsim/inertia/weight_non_lift-lbs");
		m._nWingLoad = props.globals.getNode("/fdm/jsbsim/inertia/wing_loading_lbsqft");
		
		m._cgX  	= 0;
		m._emptyCG	= 0;
		m._gross 	= 0;
		m._nonlift	= 0;
		m._wingload = 0;
		
		m._takeoff = getprop("/limits/mtow-lbs");
		
		return m;
	},
	setListeners : func(instance) {
		append(me._listeners, setlistener(weightBalance._nGrossWeight,func(n){me._onGrossWeightChange(n);},1,0) );
		append(me._listeners, setlistener(me._nCGx,func(n){me._onCGChange(n);},1,0) );
		append(me._listeners, setlistener(me._nEmptyCG,func(n){me._onEmptyCGChange(n);},1,0) );
		append(me._listeners, setlistener(me._nWingLoad,func(n){me._onWingLoadChange(n);},1,0) );
		
	},
	init : func(instance=me){
		me.setListeners(instance);
	},
	deinit : func(){
		me.removeListeners();	
	},
	_onGrossWeightChange : func(n){
		
		me._gross = me._nGross.getValue();
		me._cWeightGrossKg.setText(sprintf("%5d",me._gross/KG2LB) ~ " kg");
		me._cWeightGrossLb.setText(sprintf("%4d",me._gross) ~ " lb");
		
		if ( me._gross > 1000 ){
			me._cWeightGrossKg.setColor(COLOR["Red"]);
			me._cWeightGrossLb.setColor(COLOR["Red"]);
		}else{
			me._cWeightGrossKg.setColor(COLOR["Black"]);
			me._cWeightGrossLb.setColor(COLOR["Black"]);
		}
		
		me._nonlift = me._nNonLift.getValue();
		me._cWeightNonLiftKg.setText(sprintf( "%5d", me._nonlift * LB2KG ) ~ " kg");
		me._cWeightNonLiftKg.setText(sprintf( "%4d", me._nonlift ) ~ " lb");
		
		if ( me._nonlift > 518 ){
			me._cWeightNonLiftKg.setColor(COLOR["Red"]);
			me._cWeightNonLiftLb.setColor(COLOR["Red"]);
		}else{
			me._cWeightNonLiftKg.setColor(COLOR["Black"]);
			me._cWeightNonLiftLb.setColor(COLOR["Black"]);
		}
		
	},
	_onCGChange : func(n){
		
		me._cgX = me._nCGx.getValue();
		
		me._cCenterGravityIN.setText( sprintf("%3.1f", me._cgX) ~ " in");
		me._cCenterGravityMM.setText( sprintf("%4.1f", me._cgX * 25.4) ~ " mm");
				
		if(me._cgX * 25.4 < 240 or me._cgX * 25.4 > 360){
			me._cCenterGravityIN.setColor(COLOR["Red"]);
			me._cCenterGravityMM.setColor(COLOR["Red"]);
		}else{
			me._cCenterGravityIN.setColor(COLOR["Black"]);
			me._cCenterGravityMM.setColor(COLOR["Black"]);
		}
	},
	_onEmptyCGChange : func(n){
		
		me._emptyCG = me._nEmptyCG.getValue();
		
		me._cEmptyCGIN.setText( sprintf("%3.1f", me._emptyCG) ~ " in");
		me._cEmptyCGMM.setText( sprintf("%4.1f", me._emptyCG * 25.4) ~ " mm");
	},
	_onWingLoadChange : func(n){
		
		me._wingload = me._nWingLoad.getValue();
		
		me._cWingLoadLbSqFt.setText( sprintf("%2.1f", me._wingload) ~ " lb/sqft");
		me._cWingLoadKgSqM.setText(  sprintf("%3.1f", me._wingload * 4.8824276) ~ " kg/sqm");
	},
};

var PayloadWidget = {
	new: func(dialog,canvasGroup,name,index){
		var m = {parents:[PayloadWidget,SvgWidget.new(dialog,canvasGroup,name)]};
		m._class = "PayloadWidget";
		m._index 	= index;
		
		#debug.dump(m._listCategoryKeys);
		
		m._nRoot	= props.globals.getNode("/payload/weight["~m._index~"]");
		
		### HACK : listener on /payload/weight[0]/weight-lb not working
		###	   two props one for fdm(weight-lb) one for dialog(nt-weight-lb) listener
		m._nWeightFdm 	= m._nRoot.initNode("weight-lb",0.0,"DOUBLE");
		m._weight	= m._nWeightFdm.getValue(); # lbs
		m._nWeight 	= m._nRoot.initNode("nt-weight-lb",m._weight,"DOUBLE");
		
		m._nCapacity 	= m._nRoot.initNode("max-lb",0.0,"DOUBLE");
		
		m._capacity	= m._nCapacity.getValue();
		m._fraction	= m._weight / m._capacity;
		
		m._cFrame 		= m._group.getElementById(m._name~"_Frame");
		m._cLevel 		= m._group.getElementById(m._name~"_Level");
		m._cLB 		= m._group.getElementById(m._name~"_Lb");
		m._cKG		= m._group.getElementById(m._name~"_Kg");		
		
		m._left	= m._cFrame.get("coord[0]");
		m._right	= m._cFrame.get("coord[2]");
		m._width	= m._right - m._left;
		
		m._onWeightChange();
		
		return m;
	},
	setListeners : func(instance) {
		### FIXME : use one property remove the HACK
		append(me._listeners, setlistener(me._nWeight,func(n){me._onWeightChange(n);},1,0) );
		
		me._cFrame.addEventListener("drag",func(e){me._onInputChange(e);});
		me._cLevel.addEventListener("drag",func(e){me._onInputChange(e);});
		me._cFrame.addEventListener("wheel",func(e){me._onInputChange(e);});
		me._cLevel.addEventListener("wheel",func(e){me._onInputChange(e);});
		
	},
	init : func(instance=me){
		me.setListeners(instance);
	},
	deinit : func(){
		me.removeListeners();	
	},
	setWeight : func(weight){
		me._weight = weight;
		
		me._nWeightFdm.setValue(me._weight);
		
	},
	_onWeightChange : func( n = 0 ){
		me._weight	= me._nWeight.getValue();
		#print("PayloadWidget._onWeightChange() ... "~me._weight);
		
		me._fraction	= me._weight / me._capacity;
		
		me._cLB.setText( sprintf("%2.1f",me._weight) ~ " lb" );
		me._cKG.setText( sprintf("%2.1f",me._weight/KG2LB) ~ " kg" );
		
		me._cLevel.set("coord[2]", me._left + (me._width * me._fraction));
	},
	_onInputChange : func(e){
		var newFraction =0;
		if(e.type == "wheel"){
			newFraction = me._fraction + (e.deltaY/me._width);
		}else{
			newFraction = me._fraction + (e.deltaX/me._width);
		}
		newFraction = clamp(newFraction,0.0,1.0);
		me._weight = me._capacity * newFraction;
		
		me.setWeight(me._weight);
		
	},
};


var WaterBallastWidget = {
	new: func(dialog,canvasGroup,name,index){
		var m = {parents:[WaterBallastWidget,SvgWidget.new(dialog,canvasGroup,name)]};
		m._class = "WaterBallastWidget";
		m._index 	= index;
		
		m._nRoot	= [ props.globals.getNode("/payload/weight["~m._index[0]~"]"), props.globals.getNode("/payload/weight["~m._index[1]~"]") ];
		
		### HACK : listener on /payload/weight[0]/weight-lb not working
		###	   two props one for fdm(weight-lb) one for dialog(nt-weight-lb) listener
		m._nWeightFdm	= props.globals.getNode("/fdm/jsbsim/inertia/waterballast_total-lbs");
		m._nWeightSlider	= props.globals.getNode("/fdm/jsbsim/inertia/waterballast_total_slider-lbs");
		
		m._weight		= m._nWeightFdm.getValue();
		m._nWeight 		= [ m._nRoot[0].getNode("nt-weight-lb"), m._nRoot[1].getNode("nt-weight-lb") ];
		
		m._capacity	= 242.50849; # lbs = 110.0 kg
		m._fraction	= m._weight / m._capacity;
		
		m._cFrame 		= m._group.getElementById(m._name~"_Frame");
		m._cLevel 		= m._group.getElementById(m._name~"_Level");
		m._cLB 		= m._group.getElementById(m._name~"_Lb");
		m._cKG		= m._group.getElementById(m._name~"_Kg");		
		
		m._left	= m._cFrame.get("coord[0]");
		m._right	= m._cFrame.get("coord[2]");
		m._width	= m._right - m._left;
		
		m._onWeightChange();
		
		return m;
	},
	setListeners : func(instance) {
		### FIXME : use one property remove the HACK
		append(me._listeners, setlistener(me._nWeightFdm,func(n){me._onWeightChange(n);},1,0) );
		
		me._cFrame.addEventListener("drag",func(e){me._onInputChange(e);});
		me._cLevel.addEventListener("drag",func(e){me._onInputChange(e);});
		me._cFrame.addEventListener("wheel",func(e){me._onInputChange(e);});
		me._cLevel.addEventListener("wheel",func(e){me._onInputChange(e);});
		
	},
	init : func(instance=me){
		me.setListeners(instance);
	},
	deinit : func(){
		me.removeListeners();	
	},
	setWeight : func(weight){
		me._weight = weight;
		
		me._nWeightSlider.setValue(me._weight);
		
	},
	_onWeightChange : func( n = 0 ){
		me._weight	= me._nWeightFdm.getValue();
		#print("PayloadWidget._onWeightChange() ... "~me._weight);
		
		me._fraction	= me._weight / me._capacity;
		
		me._cLB.setText( sprintf("%2.1f",me._weight) ~ " lb" );
		me._cKG.setText( sprintf("%2.1f",me._weight/KG2LB) ~ " kg" );
		
		me._cLevel.set("coord[2]", me._left + (me._width * me._fraction));
	},
	_onInputChange : func(e){
		var newFraction =0;
		if(e.type == "wheel"){
			newFraction = me._fraction + (e.deltaY/me._width);
		}else{
			newFraction = me._fraction + (e.deltaX/me._width);
		}
		newFraction = clamp(newFraction,0.0,1.0);
		me._weight = me._capacity * newFraction;
		
		me.setWeight(me._weight);
		
	},
};

var PayloadCheckboxWidget = {
	new: func(dialog, canvasGroup, name, index, checked_property){
		var m = {parents:[PayloadCheckboxWidget,SvgWidget.new(dialog,canvasGroup,name)]};
		m._class = "PayloadCheckboxWidget";
		m._index 	= index;
		
		#debug.dump(m._listCategoryKeys);
		
		m._nRoot	= props.globals.getNode("/payload/weight["~m._index~"]");
		
		### HACK : listener on /payload/weight[0]/weight-lb not working
		###	   two props one for fdm(weight-lb) one for dialog(nt-weight-lb) listener
		m._nWeightFdm 	= m._nRoot.initNode("weight-lb",0.0,"DOUBLE");
		m._weight		= m._nWeightFdm.getValue(); # lbs
		m._nWeight 		= m._nRoot.initNode("nt-weight-lb",m._weight,"DOUBLE");
		m._nChecked		= props.globals.getNode( checked_property );
		
		m._capacity 	= m._nRoot.getNode("max-lb").getValue(); # = weight wenn checkbox is checked
		
		m._cFrame 		= m._group.getElementById(m._name~"_Frame");
		m._cChecked		= m._group.getElementById(m._name~"_Checked");
		m._cLB 		= m._group.getElementById(m._name~"_Lb");
		m._cKG		= m._group.getElementById(m._name~"_Kg");
		
		m._onWeightChange();
		m._onCheckedChange();
		
		return m;
	},
	setListeners : func(instance) {
		### FIXME : use one property remove the HACK
		append(me._listeners, setlistener(me._nWeight,func(n){me._onWeightChange(n);},1,0) );
		append(me._listeners, setlistener(me._nChecked,func(n){me._onCheckedChange(n);},1,0) );
		
		me._cFrame.addEventListener("click",func(e){me._onInputChange(e);});
		me._cChecked.addEventListener("click",func(e){me._onInputChange(e);});
		
	},
	init : func(instance=me){
		me.setListeners(instance);
	},
	deinit : func(){
		me.removeListeners();	
	},
	_onWeightChange : func(n = nil){
		me._weight	= me._nWeight.getValue();
		
		me._cLB.setText( sprintf("%2.1f",me._weight) ~ " lb" );
		me._cKG.setText( sprintf("%2.1f",me._weight/KG2LB) ~ " kg" );
	},
	_onCheckedChange : func(n = nil){
		me._cChecked.setVisible( me._nChecked.getBoolValue() );		
	},
	_onInputChange : func(e){
		me._nChecked.setBoolValue( !me._nChecked.getBoolValue() );
		me._cChecked.setVisible( me._nChecked.getBoolValue() );
	},
};

var ButtonWidget = {
	new: func(dialog, canvasGroup, name){
		var m = {parents:[ButtonWidget,SvgWidget.new(dialog,canvasGroup,name)]};
		m._class = "ButtonWidget";
		
		m._cSetDefaults	=
			[ 	m._group.getElementById("SetDefaultsBtn"),
				m._group.getElementById("SetDefaultsBtnBg"),
				m._group.getElementById("SetDefaultsBtnText")
			];
		m._cClose	=
			[ 	m._group.getElementById("CloseBtn"),
				m._group.getElementById("CloseBtnBg"),
				m._group.getElementById("CloseBtnText")
			];
		
		return m;
	},
	setListeners : func(instance) {		
		me._cSetDefaults[0].addEventListener("mouseup",func(e){me._onSetDefaultsBtn(0);});
		me._cSetDefaults[0].addEventListener("mousedown",func(e){me._onSetDefaultsBtn(1);});
		me._cClose[0].addEventListener("click",func(e){me._onCloseBtn(e);});
		
	},
	init : func(instance=me){
		me.setListeners(instance);
	},
	deinit : func(){
		me.removeListeners();	
	},
	_onSetDefaultsBtn: func( x ){
		if( x ) {
			me._cSetDefaults[1].setColorFill( 0, 0, 0 );
			me._cSetDefaults[2].setColorFill( 1, 1, 1 );
			set_defaults();
		} else {
			me._cSetDefaults[1].setColorFill( 1, 1, 1 );
			me._cSetDefaults[2].setColorFill( 0, 0, 0 );
		}
	},
	_onCloseBtn: func( e ){
		me._cClose[1].setColorFill( 0, 0, 0 );
		me._cClose[2].setColorFill( 1, 1, 1 );
		weightBalance.close();
	},
};

var WeightBalanceClass = {
	new : func(){
		var m = {parents:[WeightBalanceClass]};
		m._nRoot 	= props.globals.initNode("/asw20/dialog/weight-balance");
		
		m._nGrossWeight	= props.globals.initNode("/fdm/jsbsim/inertia/nt-weight-lbs",0.0,"DOUBLE"); #listener on weight-lbs not possible, set via system in Systems/weightbalance.xml
		
		m._name  = "Weight and Balance";
		m._title = "Weight and Balance Settings";
		m._fdmdata = {
			grosswgt : "/fdm/jsbsim/inertia/weight-lbs",
			payload  : "/payload",
			cg       : "/fdm/jsbsim/inertia/cg-x-in",
		};
		
		
		m._listeners = [];
		m._dlg 		= nil;
		m._canvas 	= nil;
		
		m._isOpen = 0;
		m._isNotInitialized = 1;
		
		m._widget = {
			Pilot	 		: nil,
			Baggage	 	: nil,
			Trim_Nose	 	: nil,
			Trim_Tail		: nil,
			Instruments	 	: nil,
			WaterBallast 	: nil,
			TailDolly	 	: nil,
			Weight		: nil,
			Buttons		: nil,
		};
		
		
		return m;
	},
	toggle : func(){
		if(me._dlg != nil){
			if (me._dlg._isOpen){
				me.close();
			}else{
				me.open();	
			}
		}else{
			me.open();
		}
	},
	close : func(){
		me._dlg.del();
		me._dlg = nil;
	},
	removeListeners  :func(){
		foreach(l;me._listeners){
			removelistener(l);
		}
		me._listeners = [];
	},
	setListeners : func(instance) {
		
		
	},
	_onClose : func(){
		me.removeListeners();
		me._dlg.del();
		
		foreach(widget;keys(me._widget)){
			if(me._widget[widget] != nil){
				me._widget[widget].deinit();
				me._widget[widget] = nil;
			}
		}
		
	},
	open : func(){
		
		me._dlg = MyWindow.new([512, 312], "dialog");
		me._dlg._onClose = func(){
			weightBalance._onClose();
		}
		me._dlg.set("title", "Weight And Balance Settings");
		me._dlg.move(100,100);
		
		
		me._canvas = me._dlg.createCanvas();
		me._canvas.set("background", "#ffffffff");
		
		me._group = me._canvas.createGroup();
		
		canvas.parsesvg(me._group, "Aircraft/asw20/gui/dialogs/WeightAndBalance.svg",{"font-mapper": weightBalanceFontMapper});
		
		me._widget.Pilot		= PayloadWidget.new(me, me._group, "Pilot", 0);
		me._widget.Baggage	= PayloadWidget.new(me, me._group, "Baggage", 1);
		me._widget.Trim_Nose	= PayloadWidget.new(me, me._group, "TrimNose", 2);
		me._widget.Trim_Tail	= PayloadWidget.new(me, me._group, "TrimTail", 3);
		me._widget.Instruments	= PayloadWidget.new(me, me._group, "Instruments", 4);
		me._widget.WaterBallast	= WaterBallastWidget.new(me, me._group, "WaterBallast", [ 5, 6 ]);
		me._widget.TailDolly	= PayloadCheckboxWidget.new(me, me._group, "TailDolly", 7, "/fdm/jsbsim/systems/groundhandling/tail-dolly");
		
		me._widget.Weight = WeightWidget.new(me,me._group,"Weight",me._widget);
		
		me._widget.Buttons	= ButtonWidget.new(me, me._group, "Buttons");
		
		foreach(widget;keys(me._widget)){
			if(me._widget[widget] != nil){
				me._widget[widget].init();
			}
		}
		
		
	},
	_onNotifyChange : func(n){
		
	},
	
};

var weightBalance = WeightBalanceClass.new();

gui.menuBind("fuel-and-payload", "dialogs.weightBalance.toggle();");

